from abc import ABCMeta, abstractmethod
from typing import List, TypeVar, Union, Type, Dict

import numpy as np

from .headers import Header, make_empty_header


class Observation:
    def __init__(self, obs_id, obi_num):
        self.obs_id = obs_id
        self.obi_num = obi_num

    def __eq__(self, other: "Observation"):
        return str(self) == str(other)

    def __str__(self):
        return f'{str(self.obs_id).zfill(5)}_{str(self.obi_num).zfill(3)}'

    def __repr__(self):
        return f'Observation: {self}'

    def __hash__(self):
        return hash(str(self))


class Stack:
    def __init__(self, stack_id):
        self.stack_id = stack_id

    def __eq__(self, other: "Stack"):
        return str(self) == str(other)

    def __str__(self):
        return str(self.stack_id)

    def __repr__(self):
        return f'Stack: {self}'


class Column(metaclass=ABCMeta):
    @property
    @abstractmethod
    def values(self):
        raise NotImplementedError()

    @values.setter
    @abstractmethod
    def values(self, values):
        raise NotImplementedError()

    @abstractmethod
    def set_variable_row_value(self, row_number, value):
        raise NotImplementedError()

    @abstractmethod
    def add_value(self, value):
        raise NotImplementedError()


class Block(metaclass=ABCMeta):
    @property
    @abstractmethod
    def observation(self) -> Observation:
        raise NotImplementedError()

    @property
    @abstractmethod
    def stack(self) -> Stack:
        raise NotImplementedError()

    def get_parent(self) -> "FitsLikeFile":
        raise NotImplementedError()

    @abstractmethod
    def write(self):
        raise NotImplementedError()

    @abstractmethod
    def get_header_value(self, name, default=None):
        raise NotImplementedError

    @abstractmethod
    def get_header_names(self):
        raise NotImplementedError

    @abstractmethod
    def get_history(self):
        raise NotImplementedError

    @abstractmethod
    def get_comments(self):
        raise NotImplementedError

    @abstractmethod
    def get_column(self, name) -> Column:
        raise NotImplementedError()

    @abstractmethod
    def new_column(self, name) -> Column:
        raise NotImplementedError()

    @property
    @abstractmethod
    def column_names(self) -> List[str]:
        raise NotImplementedError()

    @abstractmethod
    def set_metadata(self, key: str, value):
        raise NotImplementedError()

    @abstractmethod
    def update_metadata(self, metadata: Dict):
        raise NotImplementedError()

    @abstractmethod
    def get_transform(self, name):
        raise NotImplementedError()

    @property
    def columns_number(self):
        return len(self.column_names)


class FitsLikeFile(metaclass=ABCMeta):
    @abstractmethod
    def get_block(self, name: str, header_class=None) -> Block:
        raise NotImplementedError()

    @abstractmethod
    def new_block(self, name: str, data='table', header_class=None, header=None) -> Block:
        raise NotImplementedError()

    @abstractmethod
    def write(self, fname=None):
        raise NotImplementedError()

    @property
    @abstractmethod
    def blocks(self):
        raise NotImplementedError()

    @abstractmethod
    def get_caldb_version(self):
        raise NotImplementedError()


T = TypeVar('T')


class InputOutputFactory(metaclass=ABCMeta):
    @abstractmethod
    def create_file(self, filename, cls: Union[None, Type[T]] = None, clobber=False, header_class=None, header=None,
                    **kwargs) -> Union[FitsLikeFile, T]:
        raise NotImplementedError()


class ColumnProxy:
    def __init__(self, block_name, name):
        self.block_name = block_name
        self.name = name

    def __get__(self, instance: FitsLikeFile, owner):
        return instance.get_block(self.block_name).get_column(self.name).values

    def __set__(self, instance: FitsLikeFile, values):
        try:
            block = instance.get_block(self.block_name)
        except KeyError:
            block = instance.new_block(self.block_name)
        block.new_column(self.name).values = np.array(values)


class FileDecorator(FitsLikeFile):
    block_specs = {}

    def __init__(self, file: FitsLikeFile, **kwargs):
        self.file = file
        for block_name in self.block_specs:
            if block_name not in self.file.blocks:
                self.new_block(block_name, header_class=self.block_specs[block_name]['header_class'])
        for name, value in kwargs.items():
            setattr(self, name, value)

    def get_block(self, name: str, header_class=None) -> Block:
        if header_class is None and name in self.block_specs:
            header_class = self.block_specs[name]['header_class']
        return self.file.get_block(name, header_class=header_class)

    def new_block(self, name: str, data='table', header_class=None, header=None) -> Block:
        return self.file.new_block(name, data, header_class, header)

    def get_caldb_version(self):
        return self.file.get_caldb_version()

    @property
    def blocks(self):
        return self.file.blocks

    def write(self, fname=None):
        self.file.write(fname)


class AbstractBlock(Block, metaclass=ABCMeta):
    def __init__(self, header_class=None, header=None):
        defined_header_dict, undefined_header_dict = self.split_header(header_class, header)
        self.header = header_class(**defined_header_dict) if header_class is not None else make_empty_header()
        for key, value in undefined_header_dict.items():
            self.header.set_key(key, value)

    def split_header(self, header_class, header_dict):
        header_dict = header_dict or {}
        new_keys_dict = {key: value for key, value in header_dict.items() if header_class is not None
                         and key not in header_class.__parameters__}
        init_header_dict = dict(header_dict.items() - new_keys_dict.items())
        return init_header_dict, new_keys_dict

    def __getattr__(self, item):
        return getattr(self.header, item)

    def is_header_key(self, item):
        return item != "header" and self.header is not None and item in self.header.__parameters__

    def update_metadata(self, metadata):
        for key, value in metadata.items():
            self.set_metadata(key, value)

    def set_metadata(self, key: str, value):
        self.header.set_key(key, value)

    def __setattr__(self, item, value):
        if self.is_header_key(item):
            setattr(self.header, item, value)
        super().__setattr__(item, value)
