import sys
from enum import Enum
from pathlib import Path

import pytest

from ciao.tools import configuration as config
from ciao.tools import Tool
from ciao.tools.configuration.paramio_implementation import ConfigurationFactory, ParameterFileGenerator


def test_basic(monkeypatch, tmp_path):
    monkeypatch.setenv("PFILES", f'{tmp_path};{Path(__file__).parent}')
    stack_file = tmp_path / 'stack.lis'
    dummies = (tmp_path / 'file1', tmp_path / 'file2')

    with stack_file.open("a") as f:
        for dummy in dummies:
            dummy.touch()
            f.write(f'{dummy}\n')

    test_tool = ToolStub()

    config_factory = ConfigurationFactory()

    command_line = f'script'.split()
    monkeypatch.setattr(sys, 'argv', command_line)
    with pytest.raises(ValueError) as e:
        config_factory.create(test_tool)
    assert str(e.value) == "A file list instance cannot be empty (Parameter 'files')"

    command_line = f'script files=@{stack_file}'.split()
    monkeypatch.setattr(sys, 'argv', command_line)
    with pytest.raises(ValueError) as e:
        config_factory.create(test_tool)
    assert str(e.value) == "The name of an output file cannot be empty (Parameter 'outfile')"

    command_line = f'script positive_integer=10 files=@{stack_file} outfile=foo'.split()
    monkeypatch.setattr(sys, 'argv', command_line)
    with pytest.raises(ValueError) as e:
        config_factory.create(test_tool)
    assert str(e.value) == "Could not cast '' to <class 'float'> (Parameter 'non_negative_float')"

    command_line = f'script positive_integer=10 files=@{stack_file} outfile=foo non_negative_float=-1.0'.split()
    monkeypatch.setattr(sys, 'argv', command_line)
    with pytest.raises(ValueError) as e:
        config_factory.create(test_tool)
    assert str(e.value) == 'Parameter non_negative_float must be non negative'

    command_line = f'script files=@{stack_file} outfile=foo non_negative_float=1'.split()
    monkeypatch.setattr(sys, 'argv', command_line)
    configuration = config_factory.create(test_tool)
    assert configuration.positive_integer == 1000
    assert set(configuration.files) == set(dummies)
    assert len(configuration.files) == 2
    assert configuration.outfile == Path('foo')
    assert configuration.non_negative_float == 1.0
    assert configuration.category == Category.ONE
    assert configuration.normalized == 0.95

    command_line = f'script positive_integer=10 files=@{stack_file} outfile=foo non_negative_float=1 ' \
                    'category=two normalized=0.11'.split()
    monkeypatch.setattr(sys, 'argv', command_line)
    configuration = config_factory.create(test_tool)
    assert configuration.positive_integer == 10
    assert configuration.category == Category.TWO
    assert configuration.normalized == 0.11


def test_parameter_file_generator():
    generator = ParameterFileGenerator(Configuration)
    assert generator.generate() == expected_parameter_string


class Category(Enum):
    ONE = 'one'
    TWO = 'two'


class Configuration(config.Configuration):
    files = config.ListOfFiles()
    outfile = config.File(output=True)
    positive_integer = config.Positive(config.Integer(default=1000))
    non_negative_float = config.NonNegative(config.Float(default=0.0))
    category = config.Category(Category, default=Category.ONE)
    normalized = config.Normalized(default=0.95)
    false = config.Boolean(default=False)
    true = config.Boolean(default=True)


class ToolStub(Tool):
    configuration_class = Configuration

    def __init__(self):
        super().__init__(name='script', io_factory=None, configuration_factory=None)

    def _run(self, *args, **kwargs):
        pass


expected_parameter_string = """files,s,h,,,,
outfile,s,h,,,,
positive_integer,i,h,1000,,,
non_negative_float,f,h,0.0,,,
category,s,h,one,,,
normalized,f,h,0.95,,,
false,b,h,no,,,
true,b,h,yes,,,
"""