from numpy.testing import assert_array_equal

from ciao.tools.io import ColumnProxy, FitsLikeFile, FileDecorator, Header
from ciao.tools.io.adapters.crates import CratesInputOutputFactory
from ciao.tools.io.headers import StringKey


def test_create_file_with_arguments(tmp_path):
    io_factory = CratesInputOutputFactory()
    file_path = tmp_path / 'out.fits'

    pdf = io_factory.create_file(file_path, IntensityPdf,
                                 intensity=[0, 1, 2, 3], probability=[4, 5, 6, 7])
    pdf.write()

    assert_array_equal(pdf.get_block('MPDF').get_column('intensity').values, [0, 1, 2, 3])
    assert_array_equal(pdf.get_block('MPDF').get_column('MPDF').values, [4, 5, 6, 7])

    assert_array_equal(pdf.get_block('MPDF').get_column('intensity').values, pdf.intensity)
    assert_array_equal(pdf.get_block('MPDF').get_column('MPDF').values, pdf.probability)

    assert file_path.exists()

    file = io_factory.create_file(file_path)

    assert_array_equal(file.get_block('MPDF').get_column('intensity').values, [0, 1, 2, 3])
    assert_array_equal(file.get_block('MPDF').get_column('MPDF').values, [4, 5, 6, 7])
    del file

    pdf.intensity = [0, 0, 0, 0]
    pdf.write()

    file = io_factory.create_file(file_path)
    assert_array_equal(file.get_block('MPDF').get_column('intensity').values, [0, 0, 0, 0])
    assert_array_equal(file.get_block('MPDF').get_column('MPDF').values, [4, 5, 6, 7])


class Phot3PdfHeader(Header):
    origin = StringKey(default="ASC")


class IntensityPdf(FileDecorator):
    block_specs = {
        'MPDF': {'header_class': Phot3PdfHeader}
    }
    intensity = ColumnProxy(block_name='MPDF', name='intensity')
    probability = ColumnProxy(block_name='MPDF', name='MPDF')

    def __init__(self, file: FitsLikeFile, result=None, **kwargs):
        if result is not None:
            kwargs['intensity'] = result.bins
            kwargs['probability'] = result.smoothed_posterior
        backgrounds = kwargs.pop('backgrounds', None)
        super().__init__(file, **kwargs)

        self.fill_header(result, backgrounds)

    def fill_header(self, result, backgrounds):
        if result is not None:
            pdf_block = self.get_block("MPDF")
            primary_block = self.get_block("PRIMARY")

            metadata = dict(result.metadata, caldbver=self.get_caldb_version())
            pdf_block.update_metadata(metadata)
            primary_block.update_metadata(metadata)

            if backgrounds:
                for idx, background in enumerate(backgrounds.values()):
                    block = self.new_block(f"MPDF_BKG{idx+1}")
                    block.new_column("intensity").values = background.pdf.x
                    block.new_column("MPDF").values = background.pdf.y
                    block.update_metadata(background.metadata)
