from pycrates import read_file, get_keyval

from concurrent.futures import ThreadPoolExecutor
import concurrent.futures


def extract_keywords_from_file(file_path, keywords):
    """
    Extracts specified keywords from the header of a FITS file using Crates.

    Args:
    file_path (str): Path to a FITS file.
    keywords (list of str): List of keywords to extract from the FITS header.

    Returns:
    tuple: The file path and a dictionary of keyword-value pairs.
    """
    try:
        crate = read_file(file_path)
        keyword_dict = {}
        for keyword in keywords:
            value = get_keyval(crate, keyword)
            keyword_dict[keyword] = value
        return file_path, keyword_dict
    except Exception as e:
        return file_path, {'error': str(e)}


def extract_fits_keywords(file_paths, keywords, executor):
    """
    Opens a list of FITS files in parallel and extracts specified keywords from each file.

    Args:
    file_paths (list of str): List of paths to FITS files.
    keywords (list of str): List of keywords whose values need to be extracted from the FITS headers.
    executor (Executor): concurrent.futures.Executor for running tasks in parallel.
    report_func (function): Function to report progress. It should accept two integer arguments for current and total.

    Returns:
    dict: A dictionary with file paths as keys and dictionaries of keyword-value pairs as values.
    """
    results = {}
    future_to_file = {executor.submit(extract_keywords_from_file, file_path, keywords): file_path for
                      file_path in file_paths}

    for future in concurrent.futures.as_completed(future_to_file):
        file_path = future_to_file[future]
        try:
            results[file_path] = future.result()[1]
        except Exception as e:
            results[file_path] = {'error': str(e)}
    return results
