############################################################################
# 
#  Copyright (C) 2008-2011,2013,2024  Smithsonian Astrophysical Observatory
#
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License along
#  with this program; if not, write to the Free Software Foundation, Inc.,
#  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
#

#
# FILE NAME: ciao.sh
#
# DEVELOPMENT: CXCDS User Infrastructure Software
#
# DESCRIPTION:
#
#       This script configures the user environment for CIAO, the Chandra
#       X-Ray Center data analysis system.
#
#       When the script is sourced, it checks for the existence of an
#       ASCDS_INSTALL environment variable. If found, the script assumes
#       that the environment is already set up and exits with a warning
#       message. The user may override a previously set configuration by
#       setting the ASCDS_OVERRIDE environment variable (any value is
#       ok) and rerunning this script.
#
#       User customizations should be added to the 'USER CUSTOMIZATIONS'
#       section which appears slightly beyond the middle of this script.
#       If the user desires to modify the value of the installation root
#       (ASCDS_INSTALL) they should do so by setting the _nextASCDS_INSTALL
#       value at the beginning of the script.
#
#       Users should dot this file to run ". ciao.sh" to ensure that
#       the changes are written to their environment.
#
#       This version of the script is written in the Bourne shell language,
#       but is compatible with ksh and bash, too.
#
############################################################################

RemovePath()            # Remove $2 from the path variable named by $1
{
    if [ "eval '$'${1}x" != "x" ]; then
        _tmp=`eval echo '$'${1}`
        _tmp=`echo $_tmp | sed "s%$2%%g" | sed "s%::%:%g"`
        eval ${1}='$_tmp'
    fi
}

AddPathUnique()         #  Add $2 to path variable named by $1, uniquely
{
    RemovePath "${1}" "${2}"
    eval _tmp="'$'${1}"
    if [ "x${_tmp}" != "x" ]; then
	eval "${1}"="${2}:${_tmp}"
    else
        eval "${1}"="${2}"
    fi
    export ${1}
}

run_setup()
{
    if [ -f "${ASCDS_INSTALL}/bin/setup_${1}.sh" ] ; then
	. "${ASCDS_INSTALL}/bin/setup_${1}.sh"
    fi
}

################################################################
#               Setup for ASCDS Installation root              #
################################################################

_prevASCDS_INSTALL=${ASCDS_INSTALL:-""}
_nextASCDS_INSTALL=/opt/conda/envs/ciao-4.17.0
_prevCIAO_PYTHON=${CIAO_PYTHON}
if [ "$_prevASCDS_INSTALL" != "" ] ;  then
   _prevCheck=`eval echo $PATH | grep $_prevASCDS_INSTALL`
else
   _prevCheck=""
fi
_run_setup="YES"

################################################################
#                  Handle command line options                 #
################################################################
do_config=1
if [ "$ASCDS_ARGS" = "" ] ; then
   ASCDS_ARGS=$*
   _unsetArgs="YES"
else
   _unsetArgs="NO"
fi

if [ "x$ASCDS_ARGS" != "x" ] ; then
   _tmpOpt="${OPTIND}"
   # Process cmdline args
   while getopts vqoh opt ${ASCDS_ARGS} 2>/dev/null ; do
      case ${opt} in
         v) if [ "x$ASCDS_QUIET" = "x" ] ; then
               if [ "x$_prevCheck" = "x" ] || [ "$_nextASCDS_INSTALL" != "$_prevASCDS_INSTALL" ] ; then
                  if [ -f ${_nextASCDS_INSTALL}/bin/ciaover ] ; then
                     ${_nextASCDS_INSTALL}/bin/ciaover -t ${_nextASCDS_INSTALL} -v
                  else
                     echo "This script sets the user's CIAO environment to utilize:"
                     if [ -f ${_nextASCDS_INSTALL}/VERSION ] ; then
                         echo " CIAO version     : " `cat ${_nextASCDS_INSTALL}/VERSION`
                     fi
                     echo    " bin dir          : " ${_nextASCDS_INSTALL}/bin
                 fi
             fi
             if [ "x$_prevASCDS_INSTALL" != "x" ] && [ "x$_prevCheck" != "x" ] ; then
                 if [ -f $_prevASCDS_INSTALL/bin/ciaover ] ; then
                     $_prevASCDS_INSTALL/bin/ciaover -t $_prevASCDS_INSTALL -v
                 else
                     echo "The current environment is configured for:"
                     if [ -f $_prevASCDS_INSTALL/VERSION ]; then
                         echo " CIAO version     : " `cat $_prevASCDS_INSTALL/VERSION`
                     fi
                     echo   " bin dir          : " $_prevASCDS_INSTALL/bin
                 fi
             fi
          fi
          unset _prevASCDS_INSTALL
          _run_setup="NO";
          ;;
       q) ASCDS_QUIET=-q
          ;;
       o) if [ "x${ASCDS_OVERRIDE}" != "x1" ] ; then
	      ASCDS_OVERRIDE=1
	      unset_ASCDS_OVERRIDE=1
	  fi
          ;;
       h) echo "-- CIAO setup script (ksh version)"
          echo  "Usage:  .  ciao.sh [-hoqv]"
          echo "   -h      =  This help message"
          echo "   -o      =  Overide a previous configuration"
          echo "   -q      =  Quiet mode- do not echo output to screen"
          echo "   -v      =  Print ciao version information"
          _run_setup="NO"
          ;;
     esac
   done
   if [ "$_unsetArgs" = "YES" ] ; then
      unset ASCDS_ARGS
      shift $#
   fi
   OPTIND=$_tmpOpt     # preserve original OPTIND value
fi
if [ "x${_run_setup}" != "NO" ] ; then
    unset _run_setup

    if [ "$_prevASCDS_INSTALL" != "" ] ;  then
	_prevCheck=`eval echo $PATH | grep $_prevASCDS_INSTALL`
	if test "$ASCDS_OVERRIDE" || [ "$_prevCheck" = "" ]; then
	    do_config=1
	    unset PDIRS
	    RemovePath PFILES "$_prevASCDS_INSTALL/param"
	else
	    if [ "$ASCDS_QUIET" = "" ] ; then
		echo "ERROR: The current environment is configured for:"
		if [ -f ${ASCDS_INSTALL}/bin/ciaover ] ; then
		    ${ASCDS_INSTALL}/bin/ciaover -t $_prevASCDS_INSTALL
		else
		    if [ -f $_prevASCDS_INSTALL/VERSION ] ; then
			echo " CIAO version      : " `cat $_prevASCDS_INSTALL/VERSION`
		    fi
		    if [ -f $_prevASCDS_INSTALL/VERSION.prop_tk ] ; then
			echo " Proposal Toolkit version : " `cat $_prevASCDS_INSTALL/VERSION.prop_tk`
		    fi
		    echo    " bin dir          : " "$_prevASCDS_INSTALL/bin"
		    echo    " Python path      : " "$_prevCIAO_PYTHON"
		    echo "To override please set ASCDS_OVERRIDE=1 and rerun this script."
		fi
	    fi
	    do_config=0
	fi
    fi
# should we setup the ciao environment?
    if [ $do_config -eq 1 ] ; then
	
################################################################
#       Remove previous configuration values from setup        #
#       While this script no longer setups up many of these    #
#       Variables, they are still removed to provide backward  #
#       support. (i.e. Setting up CIAO in a window that old    #
#       CIAO had been started.                                 #
################################################################

	if [ "x${ASCDS_INSTALL}" != "x" ] ; then
	    RemovePath PATH "$ASCDS_INSTALL/bin"
	    RemovePath PATH "$ASCDS_INSTALL/isis/bin"
	    RemovePath PGPLOT_DIR "$ASCDS_INSTALL/pgplot"
	    RemovePath ISIS_SRCDIR "$ASCDS_INSTALL"
	    RemovePath ISIS_SRCDIR "$ASCDS_INSTALL/isis"
	    RemovePath PYTHONPATH "$ASCDS_INSTALL/lib/python3.11/site-packages"
	    
	    #      RemovePath BWIDGET_LIBRARY "$ASCDS_INSTALL/lib"
	    #      RemovePath BWIDGET_LIBRARY "$ASCDS_INSTALL/saord"
	    #      RemovePath DYLD_FRAMEWORK_PATH "$ASCDS_INSTALL/Library/Frameworks"
	fi
	
################################################################
#                set CIAO environment variables                #
################################################################
	
	ASCDS_INSTALL="$_nextASCDS_INSTALL"
	ASCDS_CONTRIB="$ASCDS_INSTALL"
	ASCDS_LIB="$ASCDS_INSTALL/lib"
	ASCDS_BIN="$ASCDS_INSTALL/bin"
	ASCDS_OTS="$ASCDS_INSTALL"
	export ASCDS_INSTALL ASCDS_LIB ASCDS_BIN ASCDS_OTS ASCDS_CONTRIB
	
	if [ "x${ASCDS_WORK_PATH}" != "x" ] ; then
	    ASCDS_DUMMY="x"
	else
		if [ "x${TMPDIR}" != "x" ] ; then
			ASCDS_WORK_PATH="${TMPDIR}"
		else
			ASCDS_WORK_PATH=/tmp
		fi
		export ASCDS_WORK_PATH
	fi
	if [ "x${ASCDS_TMP}" != "x" ] ; then
		ASCDS_DUMMY="x"
	else
		if [ "x${TMPDIR}" != "x" ] ; then
			ASCDS_TMP=${TMPDIR}
		else            
			ASCDS_TMP="/tmp"
		fi
		export ASCDS_TMP
	fi
	
	if [ -f "${ASCDS_INSTALL}/bin/ots_check" ] ; then
	    ASCDS_IMAGER_PATH="`${ASCDS_INSTALL}/bin/ots_check DS9`"
	    if [ "${ASCDS_IMAGER_PATH}" != "CIAO" ] ; then
		ASCDS_DUMMY="x"
	    else
		if [ -d "/opt/conda/envs/ciao-4.17.0/imager" ] ; then
           # This is for internal SAO/CM builds and external user configurations
		    ASCDS_IMAGER_PATH="/opt/conda/envs/ciao-4.17.0/imager"
		else
		    ASCDS_IMAGER_PATH="${ASCDS_INSTALL}/bin"
		fi
	    fi
	    export ASCDS_IMAGER_PATH
	    CIAO_XPA="`${ASCDS_INSTALL}/bin/ots_check XPA`"
	    export CIAO_XPA
	    CIAO_PYTHON="`${ASCDS_INSTALL}/bin/ots_check PYTHON`"
	    CIAO_APP_PYTHON="`${ASCDS_INSTALL}/bin/ots_check APP_PYTHON`"
	    CIAO_PYTHON_PATH="`${ASCDS_INSTALL}/bin/ots_check PYTHON_PATH`"
	    export CIAO_PYTHON CIAO_APP_PYTHON CIAO_PYTHON_PATH
	    if [ "${CIAO_PYTHON}" != "${ASCDS_INSTALL}/bin" ] && [ "${CIAO_PYTHON}" != "CIAO" ] ; then
		CIAO_IPYTHON="`${ASCDS_INSTALL}/bin/ots_check IPYTHON`"
		CIAO_APP_IPYTHON="`${ASCDS_INSTALL}/bin/ots_check APP_IPYTHON`"
		CIAO_PYTHON_EXE="${CIAO_PYTHON}/python"
	    else
		CIAO_IPYTHON="CIAO"
		CIAO_APP_IPYTHON="CIAO"
		CIAO_PYTHON_EXE="${ASCDS_INSTALL}/bin/python"
	    fi
	    export CIAO_IPYTHON CIAO_APP_IPYTHON CIAO_PYTHON_EXE
	    
       # copy .ciaorc file to Users $HOME directory if it dosn't exist.
	    if [ ! -f "${HOME}/.ciaorc" ] ; then
		\cp "${ASCDS_INSTALL}/config/system/.ciaorc" "${HOME}"
		\chmod 755 "${HOME}/.ciaorc"
	    fi
	else
            if [ -z ${ASCDS_IMAGER_PATH} ] ; then 
 	        if [ -d "/opt/conda/envs/ciao-4.17.0/imager" ] ; then
                    # This is for internal SAO/CM builds and external user configurations
		    export ASCDS_IMAGER_PATH="/opt/conda/envs/ciao-4.17.0/imager"
	        else
		    export ASCDS_IMAGER_PATH="${ASCDS_INSTALL}/bin"
	        fi
            fi
 
	    CIAO_PYTHON="$ASCDS_INSTALL/bin"
	    CIAO_IPYTHON="$ASCDS_INSTALL/bin"
	    CIAO_APP_PYTHON="$ASCDS_INSTALL/bin"
	    CIAO_APP_IPYTHON="$ASCDS_INSTALL/bin"
	    CIAO_PYTHON_PATH="prepend"
	    export CIAO_PYTHON CIAO_PYTHON_PATH CIAO_IPYTHON
	    export CIAO_APP_PYTHON CIAO_APP_IPYTHON

	    if [ -z ${CIAO_XPA} ] && [ -f /opt/conda/envs/ciao-4.17.0/bin ] ; then
	        export CIAO_XPA="/opt/conda/envs/ciao-4.17.0/bin"
	    fi
	fi
	
   # set XPA_METHOD to local to avoid issues if not network connection
	XPA_METHOD="local"
	export XPA_METHOD
	
   # CALDB environment variables
	CALDB="$ASCDS_INSTALL/CALDB"
	if [ -f "$CALDB/software/tools/caldb.config" ] ; then
	    CALDBCONFIG="$CALDB/software/tools/caldb.config"
	    CALDBALIAS="$CALDB/software/tools/alias_config.fits"
	    export CALDB CALDBCONFIG CALDBALIAS
	else
	    unset CALDB
	fi
	
   # The calibration directory path
	ASCDS_CALIB="$ASCDS_INSTALL/data"
	export ASCDS_CALIB
	
	if [ "x${ASCDS_CIAO}" != "xstandalone" ] ; then
	    ASCDS_CIAO="ciao"
	    run_setup sherpa
	    run_setup tools
	    run_setup obsvis
	fi
	
# Setup additions to PATH. ots/bin will be removed shortly
# and moved to the wrapper script.
	
################################################################
#                         Adjust paths                         #
################################################################

	AddPathUnique PATH "$ASCDS_INSTALL/bin"

################################################################
#                    Finished configuration                    #
################################################################

	if [ "$ASCDS_QUIET" = "" ] ; then
	    if [ "$ASCDS_CIAO" != "ciao" ] ; then
		ASCDS_DUMMY="x"
	    else   
		echo "CIAO configuration is complete... "
		if [ "x$_prevASCDS_INSTALL" != "x" ] && [ "x$_prevCheck" != "x" ] ; then
		    echo "The CIAO setup for this window has changed from"
		    if [ -f "$_prevASCDS_INSTALL/bin/ciaover" ] ; then
			$_prevASCDS_INSTALL/bin/ciaover -t $_prevASCDS_INSTALL
		    else
			if [ -f "$_prevASCDS_INSTALL/VERSION" ] ; then
			    echo " CIAO version     : " `cat "$_prevASCDS_INSTALL/VERSION"`
			fi
			echo    " bin dir          : " "$_prevASCDS_INSTALL/bin"
			echo    " Python path      : " "$_prevCIAO_PYTHON"
		    fi
		    echo "To:"
		fi
		if [ -f $ASCDS_INSTALL/VERSION ] ; then
		    if [ -f $ASCDS_INSTALL/bin/ciaover ] ; then
			$ASCDS_INSTALL/bin/ciaover
		    else
			echo " CIAO version     : " `cat "$ASCDS_INSTALL/VERSION"`
			echo    " bin dir          : " "$ASCDS_INSTALL/bin"
			echo    " Python path      : " "$CIAO_PYTHON"
		    fi
		fi
	    fi
	fi
    fi
fi
unset AddPathUnique _nextASCDS_INSTALL _prevASCDS_INSTALL ASCDS_DUMMY
unset _run_setup
if [ "x${unset_ASCDS_OVERRIDE}" = "x1" ] ; then
    unset ASCDS_OVERRIDE unset_ASCDS_OVERRIDE
fi
