#ifndef vm_v3math_h_INCLUDED
#define vm_v3math_h_INCLUDED

// File:   vm_v3math.h
// Author: Terry Gaetz

/* --8<--8<--8<--8<--
 *
 * Copyright (C) 2006 Smithsonian Astrophysical Observatory
 *
 * This file is part of vm_math.cvs
 *
 * vm_math.cvs is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * vm_math.cvs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the 
 *       Free Software Foundation, Inc. 
 *       51 Franklin Street, Fifth Floor
 *       Boston, MA  02110-1301, USA
 *
 * -->8-->8-->8-->8-- */

/****************************************************************************
 * Description: collection of methods for vm_V3Math
 *
 * History
 *--------
 * 0.0.0 1998-Jan-30  tjg  original version
 */

#include <vm_math/vm_vmath.h>           // vm_VMath<T,N>
#include <cstring>                      // memcpy strlen
#include <cmath>                        // sqrt fabs
#include <iostream>                     // ostream

//########################################################################
// vm_V3Math<T_fp>
//########################################################################

/** 
 * \class vm_V3Math vm_v3math.h <vm_math/vm_v3math.h>
 * 
 * A template class providing common numerical operations on 3-vectors of 
 * T_fp's (floating point type).
 * 
 * Unless otherwise noted, the operations are component by component, e.g., 
   \verbatim
       vm_V3Mathlt<float>::div_eq(v1, v2)
   \endverbatim
 * corresponds to
   \verbatim
       v1[i] /= v2[i], where i = 0,1,2.
   \endverbatim
 *
 * vm_V3Math has only static member functions; there are no data members.
 *
 * Where possible, the static member functions are inlined.
 */

template <class T_fp>
class vm_V3Math
  : public vm_VMath<T_fp,3>
{
private:

  enum ENelts_     { ENelts     = 3 };

public:

  /**
   * \typedef value_type
   *
   * A typedef for the floating point type;
   */
  typedef T_fp value_type;

  // ---------------------------------------------------------------------
  /*!
   * \defgroup vec_set_comp Set vector components
   */
  /*@{*/

  /**
   * Set components of v to x, y, z.
   *
   * @param v      vector to be set
   * @param x      x component
   * @param y      y component
   * @param z      z component
   */
  inline static void set( T_fp v[], T_fp x, T_fp y, T_fp z );

  /**
   * Set all components of v to x.
   *
   * @param v      vector to be set
   * @param x      value
   */
  inline static void set( T_fp v[], T_fp x );
  /*@}*/

  // ---------------------------------------------------------------------
  /*!
   * \defgroup vec_norm Normalize vectors
   */
  /*@{*/
  /**
   * Normalize a vector v.
   *
   * @return       return original vector magnitude
   *
   * @param v      vector to be normalized
   */
  inline static T_fp unitize( T_fp v[] );

  /**
   * Normalize a vector vi returning the normalized value in vu.
   *
   * @return       return original magnitude of vi
   *
   * @param vi     vector to be normalized
   * @param vu     normalized version of vi
   */
  inline static T_fp unitize( T_fp vu[], T_fp const vi[] );
  /*@}*/

  // ---------------------------------------------------------------------
  /*!
   * \defgroup vec_test_norm Tests for normality, orthogonality
   */
  /*@{*/
  /**
   * Test whether a vector is within tolerance tol of a unit vector
   *
   * @return       1 if v is within tolerance tol of being a unit vector,
   *               0 otherwise.
   *
   * @param v      vector to be checked
   * @param tol    tolerance
   */
  inline static int is_unit_vector( T_fp const v[], T_fp const tol );

  /**
   * Test whether a v is within tolerance tol of orthogonality to other vector
   *
   * @return       1 if v and other are orthogonal within tolerance tol,
   *               0 otherwise.
   *
   * @param v      vector to be checked
   * @param other  other vector
   * @param tol    tolerance
   */
  inline static int are_orthogonal( T_fp const v[], T_fp const other[],
                                    T_fp const tol );

  /**
   * Test whether v is within tolerance tol of orthonormality with other vector
   *
   * @return       1 if v and other are orthonormal within tolerance tol,
   *               0 otherwise.
   *
   * @param v      vector to be checked
   * @param other  other vector
   * @param tol    tolerance
   */
  inline static int are_orthonormal( T_fp const v[], T_fp const other[],
                                     T_fp const tol );
  /*@}*/

  // ---------------------------------------------------------------------
  /*!
   * \defgroup vec_dot_cross Dot and Cross products.
   */
  /*@{*/

  /**
   * Dot product of two vectors
   *
   * @return       dot (scalar) product of v1 with v2.
   *
   * @param v1     1st vector
   * @param v2     2nd vector
   */
  inline static T_fp dot( T_fp const v1[], T_fp const v2[] );

  /**
   * Cross (wedge) product of two vectors
   *
   * @return       cross (wedge) product of v1 with v2.
   *
   * @param prod   cross (wedge) product of v1 with v2.
   * @param v1     1st vector
   * @param v2     2nd vector
   */
  inline static void cross( T_fp       prod[], 
                            T_fp const v1[], T_fp const v2[] );
  /*@}*/

  // ---------------------------------------------------------------------
  /*!
   * \defgroup vec_io IO operations.
   */
  /*@{*/
  /**
   * Print a vector to an ostream.
   *
   * @param os       the ostream
   * @param v        vector to be printed
   * @param prefix   optional prefix string 
   * @param postfix  optional postfix string
   */
  inline static std::ostream&
  print_on( std::ostream& os, T_fp const v[],
            char const prefix[] = "", char const postfix[] = "" );

  /**
   * Print a vector to a FILE* stream.
   *
   * @param of       the FILE*
   * @param v        vector to be printed
   * @param prefix   optional prefix string 
   * @param postfix  optional postfix string
   */
  inline static void
  cprint_on( FILE* of, T_fp const v[],
             char const prefix[] = "", char const postfix[] = "" );
  /*@}*/

};

//########################################################################
//########################################################################
//
//    #    #    #  #          #    #    #  ######   ####
//    #    ##   #  #          #    ##   #  #       #
//    #    # #  #  #          #    # #  #  #####    ####
//    #    #  # #  #          #    #  # #  #            #
//    #    #   ##  #          #    #   ##  #       #    #
//    #    #    #  ######     #    #    #  ######   ####
//
//########################################################################
//########################################################################

//=========================================================================
template <class T_fp>
inline void vm_V3Math<T_fp>::
set( T_fp v[], T_fp x, T_fp y, T_fp z )
{ v[0] = x;  v[1] = y;  v[2] = z; }

template <class T_fp>
inline void vm_V3Math<T_fp>::
set( T_fp v[], T_fp x )
{ vm_VMath<T_fp,3>::set( v, x ); }

template <class T_fp>
inline T_fp vm_V3Math<T_fp>::
dot( T_fp const v1[], T_fp const v2[] )
{
  return v1[0] * v2[0] + v1[1] * v2[1] + v1[2] * v2[2];
}

template <class T_fp>
inline void vm_V3Math<T_fp>::
cross( T_fp result[], 
       T_fp const v1[], T_fp const v2[] )
{
  result[0] = v1[1] * v2[2] - v1[2] * v2[1];
  result[1] = v1[2] * v2[0] - v1[0] * v2[2];
  result[2] = v1[0] * v2[1] - v1[1] * v2[0];
}

template <class T_fp>
inline T_fp vm_V3Math<T_fp>::
unitize( T_fp v[] )
{
  T_fp size = sqrt( dot( v, v ) );
  if ( size == 0.0 )
  {
    return 0.0;
  }
  vm_V3Math::div_eq( v, size );

  return size;
}

template <class T_fp>
inline T_fp vm_V3Math<T_fp>::
unitize( T_fp vu[], T_fp const vi[] )
{
  vm_V3Math<T_fp>::copy( vu, vi );
  return vm_V3Math<T_fp>::unitize( vu );
}

template <class T_fp>
inline int vm_V3Math<T_fp>::
is_unit_vector( T_fp const v[], T_fp const tol )
{
  return !( (fabs(sqrt(dot(v, v))-1.0) > tol ) ? 1 : 0 );
}

template <class T_fp>
inline int vm_V3Math<T_fp>::
are_orthogonal( T_fp const v[], 
                T_fp const other[], 
                T_fp const tol )
{
  return !( (fabs(dot(v, other)) > tol ) ? 1 : 0 );
}

template <class T_fp>
inline int vm_V3Math<T_fp>::
are_orthonormal( T_fp const v[], 
                 T_fp const other[], 
                 T_fp const tol )
{
  return is_unit_vector(v,tol) && 
         is_unit_vector(other,tol) && 
         are_orthogonal(v,other,tol);
}

template <class T_fp>
inline std::ostream& vm_V3Math<T_fp>::
print_on( std::ostream& os, T_fp const v[],
          char const prefix[], char const postfix[] )
{ if ( std::strlen(prefix) ) { os << prefix; }
  os << v[0] << " " << v[1] << " " << v[2];
  if ( std::strlen(postfix) ) { os << postfix; }
  return os; }

template <class T_fp>
inline void vm_V3Math<T_fp>::
cprint_on( FILE* of, T_fp const v[],
           char const prefix[], char const postfix[] )
{ if ( std::strlen(prefix) ) { std::fprintf(of, "%s", prefix); }
  std::fprintf(of, "%.18e %.18e %.18e\n", v[0], v[1], v[2]);
  if ( std::strlen(postfix) ) { std::fprintf(of, "%s", postfix); } }

#endif  /* vm_v3math.h */
