-- --8<--8<--8<--8<--
--
-- Copyright (C) 2011 Smithsonian Astrophysical Observatory
--
-- This file is part of chandra.saotrace.aperture
--
-- chandra.saotrace.aperture is free software: you can redistribute it
-- and/or modify it under the terms of the GNU General Public License
-- as published by the Free Software Foundation, either version 3 of
-- the License, or (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.
--
-- -->8-->8-->8-->8--

local pi = require('math').pi
local aperture = require( 'saotrace.aperture' )
local require = require

module( ... )

require( 'Strict' ).restrict(_M)

--------------------------------------------------------------------------
-- conversion factors

-- degrees to radians
deg2rad  = pi / 180.0

-- arcmin to radians
arcmin2rad = pi / (60.0 * 180.0)

-- arcsec to radians
arcsec2rad = pi / (3600.0 * 180.0)


--------------------------------------------------------------------------
-- construct displacement
--
-- Inputs:
--
--   delta_X     -- X_AXAF; optical axis, +ve in direction from focal plane
--                          towards source
--   delta_Y     -- Y_AXAF; grating dispersion direction
--   delta_Z     -- Z_AXAF; SIM translation axis; +ve in anti-solar direction
--
-- returns:
--   displacement   -- table { x, y, z } in raytrace coordinates
--                     z is optical axis; +ve from source towards focal plane

function construct_displacement( delta_X, delta_Y, delta_Z )

  return { dx =  delta_Y,
	   dy = -delta_Z,
	   dz = -delta_X,
	}

end

--------------------------------------------------------------------------
-- construct disorientation
--
-- Inputs:
--
--   azmis_H        -- (arcmin) rotation by azmis_H about the -Z_H axis,
--                              taking Y_H to YP and X_H to XP; the
--                              ZP axis is the same as the Z axis.
--                              Angle is measured relative to +Y_H.
--                              Positive azmis_H rotates +Y_H towards +X_H.
--                              This is the same sense of rotation
--                              as HRMA yaw.
--   elmis_H        -- (arcmin) Rotation by elmis_H about the -YP axis
--                              (taking XP to XPP and ZP (= Z_H) to ZPP;
--                              the YPP axis is the same as the YP axis.
--                              Angle is measured relative to +Z_H.
--                              Positive azmis_H rotates +ZP towards -XP.
--                              This is the same sense as HRMA pitch.
--   clock_H        -- (deg)    Rotate by clock_H about the -XPP axis;
--                              Angle is measured relative to +Y_H.
--                              Positive clock_H rotates +YPP towards -ZPP.
--
-- returns:
--   disorientation -- table { azmis, elmis, clock } in raytrace coordinates
--                     HRMA coordinates

function construct_disorientation( azmis_H, elmis_H, clock_H )

  return {
     azmis =  arcmin2rad * azmis_H,
     elmis =  arcmin2rad * elmis_H,
     clock =  deg2rad    * clock_H,
  }

end

--------------------------------------------------------------------------
-- apply_displacement_disorientation
--
-- move a baffle according to the specified displacement/disorientation.
--
-- NOTE:  these are in raytrace coordinates.
--
-- Inputs:
--   displacement
--   disorientation
--
-- where
--
--   displacement {
--     dx              -- (mm)           decenter
--     dy              -- (mm)           decenter
--     dz              -- (mm)           despace
--   }
--   disorientation {
--     azmis           -- (radians)
--     elmis           -- (radians)
--     clock           -- (radians)
--   }

function apply_displacement_disorientation( displacement, disorientation )

  -- Motion should happen in the Body Centered Coordinate System (OSAC
  -- v7 p4-4, BCS)

  -- NOTE: angles need to be in radians for standard 'Aperture'
  -- package routines.  NOTE: distances are in mm.

  -- Variable 'motion' : function   name                performing
  -- ================    --------   ----                ----------
  --                     decenter   dx, dy            X, Y translations
  --                     despace    dz                Z translation
  --                     tilt       azmis, elmis      azmis, elmis rotations (radians)
  --                     clock      clock             clock rotation around Z (radians)

  -- Still TB checked : for the right order: OSAC requires :
  -- 1) Rotate around old OY with azmis
  -- 2) Rotate around new OX with elmis
  -- 3) Rotate around new OZ with clock  (TG)

  aperture.rotate_y( disorientation.azmis )

  --  This rotation should be around 'old' OX axis

  aperture.rotate_x( disorientation.elmis )

  --  This rotation should be around 'new' OZ axis      (clock)

  aperture.rotate_z( disorientation.clock )

  --  This translation should be along 'old' OX OY axes (decenter)
  --  and along 'old' OZ axis (despace)

  aperture.translate( displacement.dx, displacement.dy, displacement.dz )

end


--------------------------------------------------------------------------
-- construct_annulus
--
--
-- Inputs:
--
--   data -- table with info read from baffle configuration file

function construct_annulus( data )

   local types = { none = -1, transp = false, opaque = true }

   local type = types[data.annulus_type]

   if nil == type then
      error ( 'BaffleSet:construct_annulus: invalid annulus type: ' .. data.annulus_type )
   end

   local annulus_info = {}

   if ( 'none' == data.annulus_type ) then
      annulus_info = nil

   else
      annulus_info = {  ri   = data.R_i,      ro  = data.R_o,
	                dri  = data.tol_R_i,  dro = data.tol_R_o,
                        opaque = type }
   end

   return annulus_info

end

--------------------------------------------------------------------------
-- construct_struts
--
--
-- Inputs:
--
--   data -- table with info read from baffle configuration file

function construct_struts( data )

   local strut_info = {}

   if ( 'none'  == data.strut_type ) then

      strut_info = {
	 num         = 0,
	 width       = nil,
	 delta_theta = nil,
	 theta0      = nil,
	 R_i         = nil,
	 R_o         = nil,
	 type        = data.strut_type
      }

   elseif ( 'strut' == data.strut_type ) then

      strut_info = {
	 num         = data.strut_num,
	 width       = data.strut_width,
	 delta_theta = deg2rad * data.strut_dtheta,
	 theta0      = deg2rad * data.strut_theta0,
	 R_i         = nil,
	 R_o         = nil,
	 type        = data.strut_type
      }

   elseif ( 'rect' == data.strut_type ) then

      strut_info = {
	 num         = data.strut_num,
	 width       = data.strut_width,
	 delta_theta = deg2rad * data.strut_dtheta,
	 theta0      = deg2rad * data.strut_theta0,
	 R_i         = data.strut_R_i,
	 R_o         = data.strut_R_o,
	 type        = data.strut_type
      }

   else

      error ( 'BaffleSet:construct_struts: invalid strut type: ' .. data.strut_type )

   end

   return strut_info

end

--------------------------------------------------------------------------
-- do_strutpairs
--
-- do a pair of struts
--
-- Inputs:
--
-- strut_info {
--     num            --    number of struts
--     width          -- (mm)       strut width
--     R_i            -- (mm)       innermost radius for strut
--     R_o            -- (mm)       outermost radius for strut
--     delta_theta    -- (radians)  angle between strut centerlines
--     theta0         -- (radians)  offset of first strut
--     type           -- strut | rect
-- }

function do_strutpairs( strut_info )

  if strut_info and strut_info.num > 0 then

     aperture.begin_subassembly( )
     aperture.rotate_z ( strut_info.theta0 )
     local ns = strut_info.num / 2

     for n = 1, ns do
	aperture.strut( strut_info.width, true, false )
	aperture.rotate_z ( strut_info.delta_theta )
     end

     aperture.end_subassembly( )

  end

end

--------------------------------------------------------------------------
-- do_strut
--
-- do a single strut
--
-- Inputs:
--
-- strut_info {
--     num            --    number of struts
--     width          -- (mm)       strut width
--     R_i            -- (mm)       innermost radius for strut
--     R_o            -- (mm)       outermost radius for strut
--     delta_theta    -- (radians)  angle between strut centerlines
--     theta0         -- (radians)  offset of first strut
--     type           -- strut | rect
-- }

function do_strut( strut_info )

  if strut_info and strut_info.num > 0 then

    aperture.begin_subassembly( )
      aperture.rotate_z( strut_info.theta0 )
      ns = strut_info.num / 2
      n  = 0
      for n = 1, ns do

        aperture.strut( strut_info.width, true, false )
        aperture.rotate_z( strut_info.delta_theta )

      end
    aperture.end_subassembly( )

  end

end

------------------------------------------------------------------------------
-- do_rect_strut
--
-- do a rectangular strut
--
-- Inputs:
--
-- strut_info {
--     num            --    number of struts
--     width          -- (mm)       strut width
--     R_i            -- (mm)       innermost radius for strut
--     R_o            -- (mm)       outermost radius for strut
--     delta_theta    -- (radians)  angle between strut centerlines
--     theta0         -- (radians)  offset of first strut
--     type           -- strut | rect
-- }

function do_rect_strut( strut_info )

  if strut_info and  strut_info.num > 0 then

    local xcen = 0.5 * (strut_info.R_o + strut_info.R_i)
    local slen =        strut_info.R_o - strut_info.R_i

    aperture.begin_subassembly( )
      aperture.rotate_z ( strut_info.theta0 )
      aperture.translate( xcen, 0.0, 0.0 )

      for n = 1, strut_info.num do

        aperture.rect( slen, strut_info.width, true, false )
        aperture.rotate_z ( strut_info.delta_theta )

      end
    aperture.end_subassembly( )
  end

end


--------------------------------------------------------------------------
-- do_baffle
--
-- do a baffle
--
-- Inputs:
--
--   baffle       -- baffle descriptor
--   debug_baffle -- id of baffle to be debugged
--                   (or 'all' for the whole set)
--
-- where:
--
--  baffle {
--      id             --            identifier
--      z              -- (mm)       axial position of baffle
--      annulus_info   -- table      annulus data
--      strut_info     -- table      strut data
--      displacement   -- table      position    tweaks
--      disorientation -- table      orientation tweaks
--  }

function do_baffle( baffle, debug_baffle )

  aperture.begin_subassembly()

    aperture.translate( 0, 0, baffle.z)

    if  baffle.annulus_info  then

      aperture.annulus( baffle.annulus_info.ri, baffle.annulus_info.ro, false,
			baffle.annulus_info.opaque )

    end

    if  baffle.id == debug_baffle or debug_baffle == "all" then
      aperture.print_photon( 'raw_project' )
    end

    if  baffle.strut_info.type == 'strut'  then

       do_strutpairs( baffle.strut_info )

    elseif  baffle.strut_info.type == 'rect'  then

       do_rect_strut( baffle.strut_info )

    end

  aperture.end_subassembly( )

end
