#ifndef rl_Polarization_h_INCLUDED
#define rl_Polarization_h_INCLUDED

// File:   rl_Polarization.h
// Author: Terry Gaetz

/* --8<--8<--8<--8<--
 *
 * Copyright (C) 2006, 2007 Smithsonian Astrophysical Observatory
 *
 * This file is part of rl_raylib
 *
 * rl_raylib is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * rl_raylib is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the 
 *       Free Software Foundation, Inc. 
 *       51 Franklin Street, Fifth Floor
 *       Boston, MA  02110-1301, USA
 *
 * -->8-->8-->8-->8-- */

#include <dvm3/dvm3.h>                      // dvm3_Vector dvm3_RotMat
#include <rl_raylib/rl_Traits.h>            // rl_Traits::complex

class rl_ReflectionCoefPOD;                 // forward declaration

#include <cstdio>  

//########################################################################
//
// rl_PolCSPOD
//
// A class to encapuslate OSAC-style polarization amplitudes.
//
// Relation to OSAC polarization amplitudes:
//
//      c2_.q1 <--> c2comp(1),   c2_.q2 <--> c2comp(2),
//      s2_.q1 <--> s2comp(1),   s2_.q2 <--> s2comp(2)
//
//   random polarization:
//       wt =   c2_.q1.r^2 + c2_.q2.r^2 + c2_.q1.i^2 + c2_.q2.i^2
//            + s2_.q1.r^2 + s2_.q2.r^2 + s2_.q1.i^2 + s2_.q2.i^2
//
//       wt =   |c2[0]|^2 + |s2[0]|^2 + |c2[1]|^2 + |s2[1]|^2
//
//   discrete polarization:
//       wt =   c2_.q1.r^2 + c2_.q2.r^2 + c2_.q1.i^2 + c2_.q2.i^2
//            + s2_.q1.r^2 + s2_.q2.r^2 + s2_.q1.i^2 + s2_.q2.i^2
//            * 2.0 * (c2_.q1.i s2_.q1.r + c2_.q2.i s2_.q2.r
//                   - c2_.q1.r s2_.q1.i - c2_.q2.r s2_.q2.i)
//
//       wt =   |c2[0]|^2 + |s2[0]|^2 + |c2[1]|^2 + |s2[1]|^2
//            + 2 ( c2[0] s2[0]^* + c2[1] s2[1]^* )
//
//########################################################################

/** 
 * \struct rl_PolCSPOD
 *
 * A Plain Ol' Data struct (POD) encapsulating the OSAC-style
 * complex polarization amplitudes.  This is a utility POD to
 * simplify communication with BPipe.
 *
 * Relation to OSAC polarization amplitudes:
 *
 * \f$ \mathtt{c2\_.q1} \Longleftrightarrow \mathtt{c2comp(1)}, \qquad
 *     \mathtt{c2\_.q2} \Longleftrightarrow \mathtt{c2comp(2)} \f$
 * \f$ \mathtt{s2\_.q1} \Longleftrightarrow \mathtt{s2comp(1)}, \qquad
 *     \mathtt{s2\_.q2} \Longleftrightarrow \mathtt{s2comp(2)} \f$
 *
 * For random polarization:
 * \f[ \mathtt{wt} =   \mathtt{c2_.q1.r}^2 + \mathtt{c2_.q2.r}^2 
 *                   + \mathtt{c2_.q1.i}^2 + \mathtt{c2_.q2.i}^2
 *                   + \mathtt{s2_.q1.r}^2 + \mathtt{s2_.q2.r}^2 
 *                   + \mathtt{s2_.q1.i}^2 + \mathtt{s2_.q2.i}^2 \f]
 * or
 * \f[ \mathtt{wt} =  \left| \mathtt{c2}[0] \right|^2 
 *                  + \left| \mathtt{s2}[0] \right|^2 \f]
 *
 * For discrete polarization:
 * \f[ \mathtt{wt} =   \mathtt{c2_.q1.r}^2 + \mathtt{c2_.q2.r}^2 
 *                   + \mathtt{c2_.q1.i}^2 + \mathtt{c2_.q2.i}^2
 *                   + \mathtt{s2_.q1.r}^2 + \mathtt{s2_.q2.r}^2 
 *                   + \mathtt{s2_.q1.i}^2 + \mathtt{s2_.q2.i}^2
 *                   + 2 \left(\mathtt{c2_.q1.i} \mathtt{s2_.q1.r}
 *                           + \mathtt{c2_.q2.i} \mathtt{c2_.q2.r}
 *                           - \mathtt{c2_.q1.r} \mathtt{s2_.q1.i}
 *                           - \mathtt{c2_.q2.r} \mathtt{c2_.q2.i} \right) \f]
 * or
 * \f[ \mathtt{wt} =  \left| \mathtt{c2}[0] \right|^2 
 *                  + \left| \mathtt{s2}[0] \right|^2 
 *                + 2 \left( \mathtt{c2}[0] \mathtt{s2}^\ast[0]
 *                        +  \mathtt{c2}[1] \mathtt{s2}^\ast[1] \right) \f]
 * where the \f${}^\ast\f$ superscript denotes a complex conjugate.
 */

struct rl_PolCSPOD
{
  /// polarization amplitude (cosine component)
  rl_Traits::complex c2_[2];  
  /// polarization amplitude (sine   component)
  rl_Traits::complex s2_[2];  

  /**
   * Initialization method.
   *
   * @param b_over_a ratio of semiminor axis  to semi-major axis.
   * @param psi angle of the major axis from
   *       the +X axis, with psi increasing towards +Y
   */
  void init( double b_over_a = 1.0,
             double psi      = 0.0
           );

  /**
   * Attenuate intensity by a factor.
   *
   * @param factor the attenuation factor; the ray intensity is multiplied
   *        by factor.
   */
  void   attenuate( double factor );

  /**
   * Evaluate the intensity.
   *
   * @return the intensity (i.e., Stoke's \f$s_0\f$.
   */
  double intensity() const;

  /**
   * Write the polarization amplitude to stream os.
   * The state is written out as "[(r,i)(r,i)][(r,i)(r,i)]".
   *
   * @param os output stream.
   */
  std::ostream& print_on( std::ostream& os ) const;

  /**
   * Write the polarization amplitude to FILE* of.
   * The state is written out as "[(r,i)(r,i)]\n[(r,i)(r,i)]".
   *
   * @param of output FILE*.
   */
  void cprint_on( FILE* of ) const;
};

/**
 * Write the polarization amplitude to output stream os.
 * The state is written as "[(r,i)(r,i)][(r,i)(r,i)]".
 *
 * @param os output stream
 * @param polcs the rl_PolCSPOD to output.
 */
inline std::ostream&
operator<<( std::ostream& os, rl_PolCSPOD const& polcs )
  //  The state is written as "[(r,i)(r,i)][(r,i)(r,i)]".
{
  return polcs.print_on( os );
}

//########################################################################
// rl_Polarization
//########################################################################
//
/** 
 * \class rl_Polarization
 *
 * Encapsulates the polarization state of a ray.
 * The class is a simple class to handle the ray polarization state
 * resolved onto a specific coordinate system.  The polarization axes
 * are constructed from a direction vector and the coordinate axes.
 * A coordinate triple is constructed in which one axis is the direction
 * vector.
 */


class rl_Polarization
{
public:

  /// complex type
  typedef rl_Traits::complex complex;

private:

  /// real polarization ``cosine'' vector.
  dvm3_Vector C_r_;
  /// imag polarization ``cosine'' vector.
  dvm3_Vector C_i_;
  /// real polarization ``sine'' vector.
  dvm3_Vector S_r_;
  /// imag polarization ``sine'' vector.
  dvm3_Vector S_i_;

  /// PROHIBITED method: declared private and left undefined.
  rl_Polarization& operator=( rl_Polarization const& );

public:

 /** 
   * Destructor (NON-VIRTUAL)
   */
 ~rl_Polarization();

 /** 
   * Default constructor; constructs rl_Polarization with INVALID fields.
   */
  rl_Polarization();

 /** 
   * Copy constructor; constructs rl_Polarization with INVALID fields.
   *
   * @param rhs - rl_Polarization object to be copied.
   */
  rl_Polarization( rl_Polarization const& rhs );

 /** 
   * Conversion constructor.  Convert OSAC-style polarization 
   * amplitudes into polarization vector set.
   *
   * @param cs OSAC-style polarization amplitude
   * @param dir ray direction vector
   */
  rl_Polarization( rl_PolCSPOD const& cs, dvm3_Vector const& dir );

 /** 
   * Initialize the polarization state.
   * 
   * @param cs OSAC-style polarization amplitude
   * @param dir ray direction vector
   */
  void init( rl_PolCSPOD const& cs, dvm3_Vector const& dir );

 /** 
   * Evaluate rl_PolCSPOD corresponding to this polarization state.
   *
   * @param cs OSAC-style polarization amplitude
   * @param dir ray direction vector
   */
  void get_PolCSPOD( rl_PolCSPOD& cs, dvm3_Vector const& dir ) const;

 /** 
   * Evaluate intensity.
   *
   * @return intensity (i.e., Stoke's \f$s_0\f$).
   */
  double intensity() const;

 /** 
   * Rotate polarization state to frame described by rot_mtx
   *
   * @param rotated output rotated polarization state.
   *        The state is now in the frame obtained by applying rot_mtx.
   * @param rot_mtx rotation matrix.
   */
  void rotate(   rl_Polarization& rotated,   dvm3_RotMat const& rot_mtx );

 /** 
   * Rotate polarization state back from frame described by rot_mtx
   *
   * @param derotated output derotated polarization state.
   *        The state is now back in the original frame.
   * @param rot_mtx rotation matrix.
   */
  void derotate( rl_Polarization& derotated, dvm3_RotMat const& rot_mtx );

  /**
   * Attenuate reflectivity by a factor.
   * @param factor attenuation factor.
   *  (NOTE:  this multiplies the polarization components by sqrt(factor).)
   */
  void attenuate( double factor );

  /**
   * Evaluate the reflected polarization vectors.
   * Given a surface normal, incident and reflected ray direction 
   * vectors, and the parallel and perpendicular reflection coefficients,
   * evaluate the reflected polarization vectors.
   *
   * @param normal  surface normal vector.
   * @param dir_in  direction vector for incoming ray.
   * @param dir_out direction vector for reflected ray.
   * @param rflcoef complex reflection coefficients for surface.
   */
  void reflect( 
                dvm3_Vector          const& normal, 
                dvm3_Vector          const& dir_in,
                dvm3_Vector          const& dir_out,
                rl_ReflectionCoefPOD const& rflcoef 
              );

  /// return the real ``cosine' polarization vector
  dvm3_Vector const& C_r() const;
  /// return the imaginary ``cosine' polarization vector
  dvm3_Vector const& C_i() const;
  /// return the real ``sine' polarization vector
  dvm3_Vector const& S_r() const;
  /// return the imaginary ``sine' polarization vector
  dvm3_Vector const& S_i() const;

  /**
   * Write the polarization vectors to stream os.
   * The state is written as "[x y z][x y z][x y z][x y z]".
   *
   * @param os output stream.
   */
  std::ostream& print_on( std::ostream& os ) const;

  /**
   * Write the polarization amplitude to FILE* of.
   * The state is written as "[x y z][x y z][x y z][x y z]".
   *
   * @param of output FILE*.
   */
  void cprint_on( std::FILE* of ) const;
};
  
//########################################################################
//########################################################################
//
//    #    #    #  #          #    #    #  ######   ####
//    #    ##   #  #          #    ##   #  #       #
//    #    # #  #  #          #    # #  #  #####    ####
//    #    #  # #  #          #    #  # #  #            #
//    #    #   ##  #          #    #   ##  #       #    #
//    #    #    #  ######     #    #    #  ######   ####
//
//########################################################################
//########################################################################

//-------------------------------------------------------------------------
inline
rl_Polarization::
~rl_Polarization()
{}

//-------------------------------------------------------------------------
inline
rl_Polarization::
rl_Polarization()
{}

//-------------------------------------------------------------------------
inline rl_Polarization::
rl_Polarization( rl_PolCSPOD const& cs, dvm3_Vector const& dir )
{ init( cs, dir ); }

//-------------------------------------------------------------------------
inline rl_Polarization::
rl_Polarization( rl_Polarization const& rhs )
  : C_r_( rhs.C_r_ ),
    C_i_( rhs.C_i_ ),
    S_r_( rhs.S_r_ ),
    S_i_( rhs.S_i_ ) 
{}

//-------------------------------------------------------------------------
inline void rl_Polarization::
rotate( rl_Polarization& rotated, dvm3_RotMat const& rot_mtx )
{
  rot_mtx.mvmult( rotated.C_r_, C_r_ );
  rot_mtx.mvmult( rotated.C_i_, C_i_ );
  rot_mtx.mvmult( rotated.S_r_, S_r_ );
  rot_mtx.mvmult( rotated.S_i_, S_i_ );
}

//-------------------------------------------------------------------------
inline void rl_Polarization::
derotate( rl_Polarization& derotated, dvm3_RotMat const& rot_mtx )
{
  rot_mtx.mtvmult( derotated.C_r_, C_r_ );
  rot_mtx.mtvmult( derotated.C_i_, C_i_ );
  rot_mtx.mtvmult( derotated.S_r_, S_r_ );
  rot_mtx.mtvmult( derotated.S_i_, S_i_ );
}

//-------------------------------------------------------------------------
inline dvm3_Vector const& rl_Polarization::
C_r() const
{ 
  return C_r_;
}

//-------------------------------------------------------------------------
inline dvm3_Vector const& rl_Polarization::
C_i() const
{ 
  return C_i_;
}

//-------------------------------------------------------------------------
inline dvm3_Vector const& rl_Polarization::
S_r() const
{ 
  return S_r_;
}

//-------------------------------------------------------------------------
inline dvm3_Vector const& rl_Polarization::
S_i() const
{ 
  return S_i_;
}

//-------------------------------------------------------------------------
/**
 * Write the polarization vectors to output stream os.
 * The state is written as "[x y z][x y z][x y z][x y z]".
 *
 * @param os output stream
 * @param polvec the rl_Polarization vectors to output.
 */
inline std::ostream&
operator<<( std::ostream& os, rl_Polarization const& polvec )
{
  return polvec.print_on( os );
}

// rl_Polarization_h_INCLUDED
#endif
