#
#  Copyright (C) 2011, 2014, 2015, 2016, 2017, 2021
#  Smithsonian Astrophysical Observatory
#
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License along
#  with this program; if not, write to the Free Software Foundation, Inc.,
#  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
#

"""
Routines that deal with the Chandra field-of-view (FOV)
files created by the skyfov tool (and are also provided as part of
the Chandra standard-data product distribution).

"""

import os

import numpy as np

import pycrates

from ciao_contrib.logger_wrapper import initialize_module_logger

logger = initialize_module_logger("ciao_contrib.region.fov")

v1 = logger.verbose1
v3 = logger.verbose3
v4 = logger.verbose4
v5 = logger.verbose5

__all__ = (
    "FOVRegion", "AxisRange",
)


# not very robust
def _is_int(ival):
    "Are we near enough an integer for display purposes?"
    return int(ival) == ival


class AxisRange:
    """Represent an axis range in DM syntax. There's
    a minimum, maximum, and bin size (defaults to 1).
    This is intended for "physical" or "logical"
    coordinate systems, not WCS ones.

    The aim is that this is a "static" object; if you
    wish to change a value then the change_xxx routine
    will return a new object.

    The axis range is defined by three values:

      minval - the left edge of the first pixel
      maxval - the right edge of the last pixel
      size   - the width of the pixel

    size can be < 1 (but must be > 0).

    The actual min/max values of the axis may be
    adjusted from the input values to ensure

    a) the grid is aligned with the standard
    Chandra grid (that is a pixel with a lower
    edge at 0.5). This is to make it easier to
    combine ranges since all that is theoretically
    needed is a shift of an integer number of pixels,
    as long as the pixel size is the same.

    b) the last pixel is not partially filled - i.e.
    (maxval-minval) is an integral number of pixels
    of width size.

    """

    def __init__(self, minval, maxval, size=1):

        if minval >= maxval:
            raise ValueError(f"Expected minval < maxval, sent {minval} and {maxval}")

        if size <= 0:
            raise ValueError(f"Expected the size to be > 0, sent {size}")

        lo = minval
        hi = maxval

        # Use an integer if we can
        if _is_int(size):
            w = int(size)
        else:
            w = size

        # Ensure the grid is aligned with the standard Chandra
        # grid (an integer number of pixel shifts to get to 0.5).
        #
        # TODO: this isn't good for non-integer bin widths (in
        #       particular 0.1)
        #
        offset = (lo - 0.5) % w
        lo -= offset

        (n, delta) = divmod(hi - lo, w)
        if delta == 0:
            self._nbins = int(n)
        else:
            self._nbins = int(n) + 1

        self._size = w
        self._minval = lo
        self._maxval = lo + self._nbins * self._size

    @property
    def minval(self):
        "Return the lower limit of the range"
        return self._minval

    @property
    def maxval(self):
        "Return the upper limit of the range"
        return self._maxval

    @property
    def size(self):
        "Return the pixel size of the range"
        return self._size

    @property
    def nbins(self):
        "Returns the number of bins (not pixels), as an integer"
        return self._nbins

    def __repr__(self):
        return f"{self.__class__.__name__}({self.minval}, {self.maxval}, {self.size})"

    def __str__(self):
        return f"{self.minval}:{self.maxval}:{self.size}"

    def as_grid(self):
        "Return a string representation lo:hi:#nbins"

        return f"{self.minval}:{self.maxval}:#{self.nbins}"

    def copy(self):
        """Returns a copy of the object."""
        return self.__class__(self.minval, self.maxval, self.size)

    def change_minval(self, minval):
        """Copy all but the minimum value.

        Parameters
        ----------
        minval : number
           The new minimum value.

        See Also
        --------
        change_maxval, change_size

        Returns
        -------
        range : AxisRange object
            All values except for minval are taken from the current
            object.

        Notes
        -----
        This does not change the minimum value of the current object.
        """
        return self.__class__(minval, self.maxval, self.size)

    def change_maxval(self, maxval):
        """Copy all but the maximum value.

        Parameters
        ----------
        maxval : number
           The new maximum value.

        Returns
        -------
        range : AxisRange object
            All values except for maxval are taken from the current
            object.

        See Also
        --------
        change_minval, change_size

        Notes
        -----
        This does not change the maximum value of the current object.
        """
        return self.__class__(self.minval, maxval, self.size)

    def change_size(self, size):
        """Copy all but the size value.

        Parameters
        ----------
        size : number
           The new size value.

        Returns
        -------
        range : AxisRange object
            All values except for size are taken from the current
            object.

        See Also
        --------
        change_minval, change_maxval

        Notes
        -----
        This does not change the size value of the current object.
        """
        return self.__class__(self.minval, self.maxval, size)

    def get_limits(self):
        "Returns (minval, maxval)."
        return (self.minval, self.maxval)

    def union(self, other):
        """Union the ranges.

        Parameters
        ----------
        other : AxisRange-like object

        Returns
        -------
        range : AxisRange object
            The limits are the union of the current object and the
            other argument.

        See Also
        --------
        change_minval, change_maxval, change_size, intersect

        Raises
        ------
        ValueError
            If the size attributes of the current object and the other
            argument do not match.
        """

        if any([not hasattr(other, a)
                for a in ["minval", "maxval", "size"]]):
            raise TypeError("union() requires an AxisRange argument!")

        if self.size != other.size:
            raise ValueError(f"union() requires equal bin size: sent {self.size} and {other.size}")

        l1 = self.minval
        u1 = self.maxval

        l2 = other.minval
        u2 = other.maxval

        return self.__class__(min(l1, l2), max(u1, u2), self.size)

    def intersect(self, other):
        """Intersect the ranges.

        Parameters
        ----------
        other : AxisRange-like object

        Returns
        -------
        range : AxisRange object
            The limits are the union of the current object and the
            other argument.

        See Also
        --------
        change_minval, change_maxval, change_size, union

        Raises
        ------
        ValueError
            If the size attributes of the current object and the other
            argument do not match.
        """

        if any([not hasattr(other, a)
                for a in ["minval", "maxval", "size"]]):
            raise TypeError("intersect() requires an AxisRange argument!")

        if self.size != other.size:
            raise ValueError(f"intersect() requires equal bin size: sent {self.size} and {other.size}")

        if self.check_overlap(other):
            l1 = self.minval
            u1 = self.maxval

            l2 = other.minval
            u2 = other.maxval

            return self.__class__(max(l1, l2), min(u1, u2), self.size)

        return None

    def check_point(self, x):
        """Returns True if x is >= minval and < maxval."""

        return x >= self.minval and x < self.maxval

    def check_overlap(self, other, inside=False):
        """Do the two ranges overlap?

        Parameters
        ----------
        other : AxisRange-like object
        inside : bool, optional
            This argument is currently not used.

        Returns
        -------
        flag : bool
            Returns True if the two ranges overlap. Note that
            the upper edge of one range equalling the
            lower edge of the other is not considered an overlap.

        See Also
        --------
        check_contains
        """

        return other.minval < self.maxval and \
            self.minval < other.maxval

    def check_contains(self, other):
        """Does the range lie completely within the object?

        Parameters
        ----------
        other : AxisRange-like object

        Returns
        -------
        flag : bool
            Returns True if other lies within the range covered
            by the object. That is if
                other.minval >= self.minval
                other.maxval <  self.maxval

        See Also
        --------
        check_overlap

        Notes
        -----
        With the current definition,

            a.check_contains(a) == False

        This may be revised in the future.
        """

        return other.minval >= self.minval and \
            other.maxval < self.maxval


# TODO
#   - have a routine that creates region objects from the
#     coordinates
#
class FOVRegion:
    """Class used to represent the polygons in a Chandra
    FOV file that indicate the field-of-view of an
    observation.

    At the moment the data is stored in the fields of this
    object; this may change in the future.

    """

    def __init__(self, fov, pixsize=1):
        """Read in the data from the file or crate
        fov.

        An IOError is thrown if the input does not contain
        the necessary information.

        pixsize is the pixelsize to use for the axis grids;
        it must be > 0
        """

        if isinstance(fov, pycrates.TABLECrate):
            cr = fov

            # Unfortunately we lose any filter applied to the
            # filename by the caller using this approach.
            #
            fname = cr.get_filename()

        else:
            cr = pycrates.TABLECrate(fov, mode="r")
            fname = fov

        if pixsize <= 0:
            raise ValueError(f"pixsize must be > 0, not {pixsize}")

        self.filename = os.path.abspath(fname)
        self.pixsize  = pixsize
        self._read_input(cr)

    def __repr__(self):
        out = f"{self.__class__.__name__}('{self.filename}'"
        if self.pixsize != 1:
            out += f",pixsize={self.pixsize}"

        out += ")"
        return out

    def __str__(self):
        out = [f"FOV      : {self.filename}",
               f"Obs Id   : {self.obsid}",
               f"Detector : {self.detector}"
               ]
        out.extend([f"CCD {c}    : X={self.ranges[c][0]} Y={self.ranges[c][1]}"
                    for c in self.ccd_ids])

        if len(self.ccd_ids) > 1:
            out.append(f"Combined : X={self.range[0]} Y={self.range[1]}")

        return "\n".join(out)

    def _read_input(self, cr):
        """Reads in the needed data from the supplied crate.

        Raises an IOError if the input does not contain the
        necessary columns and keywords to represent a Chandra fov file,
        or if there is no region that can be handled (rows that
        can not be handled are skipped).
        """

        fname = self.filename
        pixsize = self.pixsize

        v3(f"Validating input from {fname}")

        if cr.get_nrows() < 1:
            raise IOError(f"No data found in {fname}")

        # We always check for EQPOS, even though this may not be
        # required by the user.
        cnames = ["CCD_ID", "POS", "EQPOS", "SHAPE"]
        for cname in cnames:
            if not cr.column_exists(cname):
                raise IOError(f"Column {cname} not found in {fname}")

        v3("Columns exist")

        keys = ["OBS_ID", "INSTRUME", "DETNAM"]
        for key in keys:
            if not cr.key_exists(key):
                raise IOError(f"Keyword {key} not found in {fname}")

        v3("Keys exist")

        shapes = cr.get_column("SHAPE").values
        idx, = np.where(shapes == "Polygon")
        if len(idx) == 0:
            raise IOError(f"No Polygons found in {fname}")

        self.ccd_ids = cr.get_column("CCD_ID").values[idx].copy()
        pos   = cr.get_column("POS").values[idx].copy()
        eqpos = cr.get_column("EQPOS").values[idx].copy()

        # Clean up the regions to remove the trailing "blank" entries.
        # We look for the repeated point (ie where x[i] == x[0] and
        # y[i] == y[0]) using the pos column.
        #
        def getidx(rnum):
            x = pos[rnum][0]
            y = pos[rnum][1]
            idx = ((x == x[0]) & (y == y[0])).nonzero()[0]
            nidx = len(idx)
            if nidx < 2:
                raise IOError(f"Unable to find end of region for ccd_id={self.ccd_ids[rnum]}")

            # Use the last point that is equal (in case there are
            # multiple occurrences, which is highly unlikely)
            #
            return idx[-1]

        self.pos = {}
        self.eqpos = {}
        self.ranges = {}
        xr = None
        yr = None

        # Ugly code
        #   - the pos_min/max values are not
        #     really needed now we have AxisRange entities
        #     for each chip
        #
        self.pos_min = [None, None]
        self.pos_max = [None, None]
        self.eqpos_min = [None, None]
        self.eqpos_max = [None, None]

        for i in range(0, len(self.ccd_ids)):

            ccd = self.ccd_ids[i]
            v3(f"Cleaning up polygon data for ccd {ccd}")
            v5(f"x = {pos[i][0]}")
            v5(f"y = {pos[i][1]}")
            ns = getidx(i)

            v4(f"Cut at element {ns}")

            self.pos[ccd] = pos[i, :, :ns].copy()
            self.eqpos[ccd] = eqpos[i, :, :ns].copy()

            v5(f"-> x = {self.pos[ccd][0]}")
            v5(f"-> y = {self.pos[ccd][1]}")

            # some repeated work with the *_min/max fields below
            #
            xmin = self.pos[ccd][0].min()
            xmax = self.pos[ccd][0].max()
            ymin = self.pos[ccd][1].min()
            ymax = self.pos[ccd][1].max()
            x = AxisRange(xmin, xmax, pixsize)
            y = AxisRange(ymin, ymax, pixsize)
            self.ranges[ccd] = (x, y)
            if xr is None:
                xr = x
                yr = y
            else:
                xr = xr.union(x)
                yr = yr.union(y)

            for j in [0, 1]:
                if self.pos_min[j] is None or \
                        self.pos_min[j] > self.pos[ccd][j].min():
                    self.pos_min[j] = self.pos[ccd][j].min()
                if self.eqpos_min[j] is None or \
                        self.eqpos_min[j] > self.eqpos[ccd][j].min():
                    self.eqpos_min[j] = self.eqpos[ccd][j].min()

                if self.pos_max[j] is None or \
                        self.pos_max[j] < self.pos[ccd][j].max():
                    self.pos_max[j] = self.pos[ccd][j].max()
                if self.eqpos_max[j] is None or \
                        self.eqpos_max[j] < self.eqpos[ccd][j].max():
                    self.eqpos_max[j] = self.eqpos[ccd][j].max()

        self.pos_min = tuple(self.pos_min)
        self.pos_max = tuple(self.pos_max)
        self.eqpos_min = tuple(self.eqpos_min)
        self.eqpos_max = tuple(self.eqpos_max)

        self.range = (xr, yr)

        self.obsid = cr.get_key_value("OBS_ID")
        self.detector = cr.get_key_value("DETNAM")
        self.instrument = cr.get_key_value("INSTRUME")

        try:
            self.eqpos_transform = cr.get_transform("EQPOS").copy()
        except ValueError:
            self.eqpos_transform = None

        # let the user know about unsupported rows
        #
        if len(idx) != cr.get_nrows():
            idx, = np.where(shapes != "Polygon")
            bad = [str(d)
                   for d in cr.get_column("CCD_ID").values[idx]]
            wmsg = "WARNING: Unable to read regions for CCDs " + \
                f"{', '.join(bad)} in obsid {self.obsid}"
            v1(wmsg)
