import pytest

from ..process_results import process_results


def test_process_results():
    # Mock data containing various DS_IDENT formats
    results = {
        '/file1.fits': {'DS_IDENT': '10.25574/csc2.obs.17145'},
        '/file2.fits': {'DS_IDENT': '10.25574/csc2.stk.17513'},
        '/file3.fits': {'DS_IDENT': '10.25574/17506'},
        '/file4.fits': {'DS_IDENT': 'invalid_format'},
        '/file5.fits': {'DS_IDENT': '10.25574/csc2.obs.17140'},
        '/file6.fits': {'DS_IDENT': '10.25574/csc2.stk.19989'},
        '/file7.fits': {}  # No DS_IDENT
    }

    expected_results = {
        "CSC ObsID": {'17145', '17140'},
        "CSC Stack Detect ID": {'17513', '19989'},
        "Archival ObsID": {'17506'},
        "Warning - These IDs are not DOIs": {'invalid_format'}
    }

    # Call the function with the mock data
    actual_results = process_results(results)

    # Assertions to check if the function outputs the expected results
    assert actual_results['CSC ObsID'] == expected_results['CSC ObsID']
    assert actual_results['CSC Stack Detect ID'] == expected_results['CSC Stack Detect ID']
    assert actual_results['Archival ObsID'] == expected_results['Archival ObsID']
    assert actual_results['Warning - These IDs are not DOIs'] == expected_results['Warning - These IDs are not DOIs']


@pytest.mark.parametrize("ds_ident, expected_set", [
    ('10.25574/csc2.obs.17145', 'CSC ObsID'),
    ('10.25574/csc2.stk.17513', 'CSC Stack Detect ID'),
    ('10.25574/17506', 'Archival ObsID'),
    ('invalid_format', 'Warning - These IDs are not DOIs'),
    ('10.25574/csc2.obs.19989', 'CSC ObsID'),
    ('10.25574/csc2.stk.17140', 'CSC Stack Detect ID'),
])
def test_individual_identifiers(ds_ident, expected_set):
    # Test each DS_IDENT independently to ensure it is classified correctly
    results = {'/test.fits': {'DS_IDENT': ds_ident}}
    output = process_results(results)
    assert ds_ident.split('.')[-1].split('/')[-1] in output[expected_set]
