local assert = assert
local error = error
local table = require( 'table' )
local string = require( 'string' )
local tonumber = tonumber

local math = require('math')
local sqrt = math.sqrt
local pi   = math.pi
local cos  = math.cos
local sin  = math.sin

local coroutine = require('coroutine')
local yield = coroutine.yield

local ea   = require('saotrace.raygen.ea')
local vobj = require('saotrace.raygen.validate').vobj:new()

local RDB = require( 'rdb' )
local tables = require('saotrace.suplib.tables')

local M = {}
setfenv( 1, M )

vspec = {}


local function r_equal_area( rmin, rmax, n )

   local r = {}

   local rmin2 = rmin^2
   local area = rmax^2 - rmin2

   n = n - 1
   for i = 0, n do

      r[i+1] = sqrt( i/n * area + rmin2 )

   end

   return r

end


local function generate_ringNspoke( args )

   local deg2rad = pi / 180

   local ring_rad = {}

   if  args.dist == 'uniform' then

      if args.nrings == 1 then

	 ring_rad = { (args.rmax + args.rmin) / 2 }

      elseif args.nrings == 2 then

	 ring_rad = { args.rmin, args.rmax }

      else

	 local rdelta = ( args.rmax - args.rmin  ) / ( args.nrings - 1 )

	 for ring = 1, args.nrings do
	    ring_rad[ring] = args.rmin + rdelta * ( ring - 1 )
	 end

      end

   elseif args.dist == 'area' then

      if args.nrings == 1 then

	 local rt = r_equal_area( args.rmin, args.rmax, 3 )

	 ring_rad = { rt[2] }

      elseif args.nrings == 2 then

	 ring_rad = { args.rmin, args.rmax }

      else

	 ring_rad = r_equal_area( args.rmin, args.rmax, args.nrings )

      end


   else

      error( "internal error: unknown ring distribution: " .. args.dist )

   end


   local tmin = args.tmin
   local tdelta

   -- if bounds are the same, don't overlap spokes at the bounds
   if (args.tmax - args.tmin) % 360 == 0 then

      if args.nspokes == 1 then

	 tdelta = 0

      elseif args.nspokes == 2 then

	 tdelta = ( args.tmax - args.tmin ) / 2

      else

	 tdelta = ( args.tmax - args.tmin  ) / args.nspokes

      end


   else

      if args.nspokes == 1 then

	 tmin = (args.tmax + args.tmin) / 2
	 tdelta = 0

      elseif args.nspokes == 2 then

	 tdelta = args.tmax - args.tmin

      else

	 tdelta = ( args.tmax - args.tmin  ) / ( args.nspokes - 1 )

      end

   end

   for ring = 1, args.nrings do

      local r = ring_rad[ring]

      for spoke = 1, args.nspokes do

	 local theta = args.trot + tmin + ( spoke - 1 ) * tdelta;
	 local theta_rad = theta * deg2rad

	 yield{
	    spoke = spoke,
	    ring = ring,
	    r = r,
	    theta = theta,
	    x = args.x + r * cos(theta_rad),
	    y = args.y + r * sin(theta_rad),
	    z = args.z,
	    area = args.area,
	 }

      end

   end

end

vspec.ringNspoke = {
   x       = { type = 'number', default = 0 },
   y       = { type = 'number', default = 0 },
   trot    = { type = 'number', default = 0 },
   tmin    = { type = 'number', default = 0 },
   tmax    = { type = 'number', default = 360 },
   nspokes = { type = 'posint' },
   nrings  = { type = 'posint' },
   rmin    = { type = 'zposnum' },
   rmax    = { type = 'posnum' },
   dist    = { enum = { 'uniform', 'area' }, default = 'area' },
}

function ringNspoke( ... )

   local ok, args = vobj:validate(vspec.ringNspoke, ... )
   assert(ok, args)

   local co = coroutine.create( function () generate_ringNspoke( args ) end )
   return function()
	     local ok, res = coroutine.resume( co )
	     if not ok then
		error( res )
	     end

	     return res
	  end
end


vspec.ring_n_spoke = {

   file    = { type = 'string'                },
   match   = { type = { 'function', 'table' } },
   col     = { type = 'table',  default = {}   },
   format  = { enum = 'rdb',    default = 'rdb'  },
   nspokes = vspec.ringNspoke.nspokes,
   nrings  = vspec.ringNspoke.nrings,
   tmin    = vspec.ringNspoke.tmin,
   tmax    = vspec.ringNspoke.tmax,
   dist    = vspec.ringNspoke.dist,
   mi      = { type = 'number', default = 0   },
   mo      = { type = 'number', default = 0   },
   area    = { type = 'posnum', default = 1   },
   prefix  = { type = 'posnum', default = 1   },
   mode    = { enum = { 'function', 'iterator', 'coroutine', 'callback' },
	       default = 'function' },
   callback = { type = 'function', optional = true },
}

function ring_n_spoke( ... )

   local ok, args = vobj:validate( vspec.ring_n_spoke, ... )

   assert( ok, args )

   if args.mode == 'iterator' then

      args.mode = 'coroutine'
      local co = coroutine.create( function() ring_n_spoke( args ) end)
      return function()
		local res = { coroutine.resume( co ) }
		if not res[1] then
		   error( unpack( res, 2  ) )
		end

		return unpack( res, 2 )
	     end
   elseif args.mode == 'callback' and args.callback == nil then

      error( 'must specify callback function of mode is callback' )

   end

   local pars = { 'ri', 'ro', 'x', 'y', 'z' }

   local matches = {}
   local nmatches = 0


   local select

   if args.format == 'rdb' then

      select = RDB.select{ file = args.file,
			   match = args.match,
			   mode = 'iterator' }

   end

   local entries = {}
   local nentries = 0
   for rec, idx in select do

      nmatches = nmatches + 1

      local tag = table.concat( { args.file, idx }, ':' )
      local data = tables.extract( rec, pars, args.col )

      local pars = {
	 x = data.x,
	 y = data.y,
	 rmax = data.ro + args.mo,
	 rmin = data.ri + args.mi,
	 tmin = args.tmin,
	 tmax = args.tmax,
	 nrings = args.nrings,
	 nspokes = args.nspokes,
	 dist = args.dist
      }

      nspot = 1
      for spot in ringNspoke( pars ) do

	 local id  = string.format( "%d%02d%03d%03d",
				    args.prefix, nmatches,
				    spot.ring, spot.spoke )
	 local tag = string.format( "%s:[%03d][%03d]",
				    tag, spot.ring, spot.spoke )

	 tag = table.concat( { tag, id }, ':' )

	 local result = { nmatches = nmatches,
			  match = data,
			  nspot = nspot,
			  spot = spot,
			  tag = tag,
			  id = tonumber(id)
		       }

	 if args.mode == 'coroutine' then

	    yield( result )

	 elseif args.mode == 'callback' then

	    args.callback( result )

	 else

	    table.insert( entries, result )

	 end

	 nentries = nentries + 1

      end

   end

   if args.mode == 'function' then

      return entries

   elseif args.mode == 'callback' then

      return nentries

   end

end

return M
