# 
#  Copyright (C) 2010  Smithsonian Astrophysical Observatory
#
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License along
#  with this program; if not, write to the Free Software Foundation, Inc.,
#  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
#

from cxcdm import *

# ------------------------------------------------------------------------------
#                              dmDatasetAdvanceBlocks                           
# ------------------------------------------------------------------------------
def dmDatasetAdvanceBlocks( ds, nblocks ):
    """
    dmDatasetAdvanceBlocks( ds=<dmDataset>, nblocks=# )

    Open the block in the provided dataset which is nblocks ahead
    of the current block.

    Returns dmBlock object on success
    Thows RuntimeError exception on failure to open

       ds - Required
          Dataset from which to open block.

       nblocks - Required
          Number of blocks to move.

    """
    res = None

    try:
        cur = dmDatasetGetCurrentBlockNo( ds )
        blockNo = cur + nblocks

        res = dmDatasetMoveToBlock( ds, blockNo )
    except:
        raise

    return res

# ------------------------------------------------------------------------------
#                              dmDatasetNextBlock                               
# ------------------------------------------------------------------------------
def dmDatasetNextBlock( ds ):
    """
    dmDatasetNextBlock( ds=<dmDataset> )

    Move to the next block in the dataset.

    Returns dmBlock object on success
    Thows RuntimeError exception on failure to open

       ds - Required
          Dataset from which to open block.
    """
    res = None

    try:
        res = dmDatasetAdvanceBlocks( ds, 1 )
    except:
        raise

    return res


# ------------------------------------------------------------------------------
#                              dmDatasetPrint                                   
# ------------------------------------------------------------------------------
def dmDatasetPrint( ds ):
    """
    tstr = dmDatasetPrint( ds=<dmDataset> )

    Display contents of the dmDataset
    Returns a formatted string.

    """
    res = ""

    if type( ds ) is not dmDataset:
        raise TypeError("dmDatasetPrint() argument 1 must be type dmDataset.")

    try:
        if dmDatasetWriteable( ds ):
            mode = 'Write'
        else:
            mode = 'Read Only'

        nblocks  = dmDatasetGetNoBlocks( ds )
        curblock = dmDatasetGetCurrentBlockNo(ds)

        res += "  ----------------------------------------------------------------------\n"
        res += "  Dataset: %s\n" % dmDatasetGetName( ds )
        res += "        mode = %s\n" % mode
        res += "     nblocks = %d\n" % nblocks
        res += "     default = %d\n" % curblock
        res += "  ----------------------------------------------------------------------\n"
        res += "       Block  Name                Type                                  \n"
        res += "  ----------------------------------------------------------------------\n"

        for ii in range(1,nblocks+1):
            try:
                bname = dmDatasetGetBlockName( ds, ii )
                btype = dmDatasetGetBlockType( ds, ii )
            except:
                raise
            res += "  Block %4d: %-20s %-10s\n" % (ii, bname, btype)

    except:
        raise

    return res
