import pytest

from ciao.tools.io import FileDecorator
from ciao.tools.io.headers import Header, IntegerKey, FloatKey


def test_file_creation(io_factory, tmp_path):
    class HeaderUnderTest(Header):
        int_key = IntegerKey(default=99)

    test_file = io_factory.create_file(filename=tmp_path/"foo.fits")
    test_file.new_block('a_block', header_class=HeaderUnderTest)

    assert test_file.get_block('a_block').int_key == 99


def test_file_creation_with_header(io_factory, tmp_path):
    class HeaderUnderTest(Header):
        float_key = FloatKey()
        int_key = IntegerKey(default=99)

    with pytest.raises(TypeError) as e:
        test_file = io_factory.create_file(filename=tmp_path/"foo.fits")
        test_file.new_block('a_block', header_class=HeaderUnderTest)

    assert str(e.value) == "missing a required argument: 'float_key'"

    test_file = io_factory.create_file(filename=tmp_path/"foo.fits")
    test_file.new_block(name="a_block", header_class=HeaderUnderTest, header={'float_key': 3.1})
    block = test_file.get_block("a_block")
    assert block.float_key == 3.1

    test_file = io_factory.create_file(filename=tmp_path/"foo.fits")
    test_file.new_block(name="a_block", header_class=HeaderUnderTest, header={'float_key': 3.1, 'int_key': 1})
    block = test_file.get_block("a_block")
    assert block.float_key == 3.1
    assert block.int_key == 1


def test_creation_with_undefined_key(io_factory, tmp_path):
    class HeaderUnderTest(Header):
        int_key = IntegerKey(default=99)

    test_file = io_factory.create_file(filename=tmp_path/"foo.fits")
    block = test_file.new_block("a_block", header_class=HeaderUnderTest, header={'new_key': "foo"})

    assert block.new_key == "foo"


def test_validation(io_factory, tmp_path):
    class HeaderUnderTest(Header):
        int_key = IntegerKey(default=99)

    test_file = io_factory.create_file(filename=tmp_path/"foo.fits")
    block = test_file.new_block("a_block", header_class=HeaderUnderTest)

    with pytest.raises(ValueError) as e:
        block.int_key = "foo"

    assert str(e.value) == "Could not cast 'foo' to <class 'int'> (Parameter 'int_key')"


def test_round_trip(io_factory, tmp_path):
    class HeaderUnderTest(Header):
        int_key = IntegerKey(default=99)

    test_file = io_factory.create_file(filename=tmp_path/"foo.fits")
    test_file.new_block("a_block", header_class=HeaderUnderTest, header={'new_key': "foo", 'int_key': 3})

    test_file.write()

    del test_file

    test_file = io_factory.create_file(filename=tmp_path/"foo.fits")
    block = test_file.get_block("a_block", header_class=HeaderUnderTest)
    assert block.int_key == 3
    assert block.new_key == "foo"


def test_read_with_default_header(io_factory, tmp_path):
    class HeaderUnderTest(Header):
        int_key = IntegerKey(default=99)

    test_file = io_factory.create_file(filename=tmp_path / "foo.fits")
    test_file.new_block("PRIMARY", header_class=HeaderUnderTest, header={'new_key': "foo", 'int_key': 3})

    test_file.write()

    del test_file

    class FileSpec(FileDecorator):
        block_specs = {
            'PRIMARY': {'header_class': HeaderUnderTest}
        }

    test_file = io_factory.create_file(filename=tmp_path / "foo.fits", cls=FileSpec)
    block = test_file.get_block("PRIMARY")
    assert block.int_key == 3


def test_multiple_get_blocks(io_factory, tmp_path):
    class HeaderUnderTest(Header):
        int_key = IntegerKey(default=99)
        other_int = IntegerKey(default=100)

    class FileSpec(FileDecorator):
        block_specs = {
            'PRIMARY': {'header_class': HeaderUnderTest}
        }

    test_file = io_factory.create_file(filename=tmp_path / "foo.fits", cls=FileSpec)

    test_file.get_block("PRIMARY").int_key = 3
    test_file.get_block("PRIMARY").other_int = 4

    block = test_file.get_block("PRIMARY")

    assert block.other_int == 4
    assert block.int_key == 3
