############################################################################
# 
#  Copyright (C) 2008-2011,2017,2020,2022-2024  Smithsonian Astrophysical Observatory
#
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License along
#  with this program; if not, write to the Free Software Foundation, Inc.,
#  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
#

#
# FILE NAME: ciao.csh
#
# DEVELOPMENT: CXCDS User Infrastructure Software
#
# DESCRIPTION:
#
#       This script configures the environment for Chandra Science Center
#       end-user (ie, external to the CXCDS) software.
#
#       When the script is sourced, it checks for the existence of an
#       ASCDS_INSTALL environment variable. If found, the script assumes
#       that the environment is already set up and exits with a warning
#       message. Using the '-o' option will cause the script to reconfigure
#       to run using its values. The previous environment variables will
#       generally be replaced and the old install path will be removed 
#       from the user's PATH. 
#
#       User customizations should be added to the 'USER CUSTOMIZATIONS'
#       section which appears slightly beyond the middle of the script. 
#       If the user desires to modify the value of the installation root
#       (ASCDS_INSTALL) they should do so by setting the _nextASCDS_INSTALL
#       value at the beginning of the script.
#
#       This version of the script is written in the csh language, and
#       is compatible tcsh, too.
#
############################################################################

################################################################
#               Setup for ASCDS Installation root              #
################################################################

set _silent = "No"
set _overrideNeeded = "No" 
set _partialConfig = "No"
if ( $?ASCDS_INSTALL ) then 
   set _prevASCDS_INSTALL = $ASCDS_INSTALL 
   if (`echo $PATH | grep $ASCDS_INSTALL` != "" ) then
      set _overrideNeeded = "Yes"
   else
      set _partialConfig = "Yes"
   endif
   if ( $?CIAO_PYTHON ) then
       set _prevCIAO_PYTHON = "$CIAO_PYTHON"
   else
       set _prevCIAO_PYTHON = ""
   endif
else
   set _prevASCDS_INSTALL =  ""
   set _prevCIAO_PYTHON = ""
endif 
set _nextASCDS_INSTALL = "/opt/conda/envs/ciao-4.17.0"
set _nextCIAO_PYTHON = "$_nextASCDS_INSTALL/bin/python"

################################################################
#                  Handle command line options                 #
################################################################
set arg_cnt=1
set nargs=$#argv
foreach value ($argv[*])
   switch ( $argv[$arg_cnt] )
      case -h
         echo "-- CIAO setup script (csh version)"
         echo  "Usage:  source ciao.csh [-hoOqv]"
         echo "   -h       =  This help message"
         echo "   -o       =  Overide a previous configuration"
         echo "   -q       =  quiet mode- do not echo output to screen"
         echo "   -O <DIR> =  Override default ASCDS_INSTALL location with DIR"
	 echo "               (for internal CXC use, in csh/tcsh only)"
         echo "   -v       =  Print CIAO version information"
         exit
      breaksw
      case -v
         if ( $_silent == "No" ) then 
            if (( $_overrideNeeded == "No") || \
                 ( "$_nextASCDS_INSTALL" != "$_prevASCDS_INSTALL" )) then
	       if ( -e "$_nextASCDS_INSTALL/bin/ciaover" ) then
		   $_nextASCDS_INSTALL/bin/ciaover -t $_nextASCDS_INSTALL -v
	       else
                  echo "This script sets the user's CIAO environment to utilize:"
                  if ( -e "$_nextASCDS_INSTALL/VERSION" ) then
                      echo " CIAO version     : " `cat $_nextASCDS_INSTALL/VERSION`
		      if ( -e "$_nextASCDS_INSTALL/VERSION.prop_tk" ) then
			  echo " Proposal Toolkit version : " `cat $_nextASCDS_INSTALL/VERSION.prop_tk`
		      endif
		      echo    " bin dir          : " "$_nextASCDS_INSTALL/bin"
#	              echo    " Python path      : " "$_nextCIAO_PYTHON"
		   endif
	       endif
            endif
            if ( $_overrideNeeded == "Yes" ) then
	        if ( -e "$_prevASCDS_INSTALL/bin/ciaover" ) then
		   $_prevASCDS_INSTALL/bin/ciaover -t $_prevASCDS_INSTALL -v
	        else
                   echo "The current environment is configured for:"
                   if ( -e "$_prevASCDS_INSTALL/VERSION" ) then
                      echo " CIAO version     : " `cat $_prevASCDS_INSTALL/VERSION`
                   endif
                   if ( -e "$_prevASCDS_INSTALL/VERSION.prop_tk" ) then
                      echo " Proposal Toolkit version : " `cat $_prevASCDS_INSTALL/VERSION.prop_tk`
                   endif
                   echo    " bin dir          : " "$_prevASCDS_INSTALL/bin"
#	           echo    " Python path      : " "$_prevCIAO_PYTHON"
		endif
            endif
         endif
         exit
      breaksw
      case -o
         set _overrideNeeded = "No"
      breaksw
      case -q
         set _silent = "-q"
      breaksw
      case -O
	 @ arg_cnt++
	 set _nextASCDS_INSTALL = $argv[$arg_cnt]
      breaksw
   endsw
   if ($arg_cnt == $nargs) then
      break
   endif
   @ arg_cnt++ 
end

if ( $_overrideNeeded == "No" ) then 
   if ( $_prevASCDS_INSTALL != "" ) then
      unsetenv PDIRS
      # Remove previous system param file from $PFILES if 'override'
      if ( $?PFILES ) then
	if ( `echo "$PFILES" | grep "$_prevASCDS_INSTALL/param"` != "" ) then
          setenv PFILES `echo "{$PFILES}" | sed "s%$_prevASCDS_INSTALL/param%%g"`
	  setenv PFILES `echo "{$PFILES}" | sed "s%::%:%g"`
	endif
      endif
   endif
else
   if ( $_silent == "No" ) then 
      echo "ERROR: The current environment is configured for:"
      if ( -e "$_prevASCDS_INSTALL/VERSION" ) then
	 if ( -e "$_prevASCDS_INSTALL/bin/ciaover" ) then
	    $_prevASCDS_INSTALL/bin/ciaover -t $_prevASCDS_INSTALL
	 else
	    echo " CIAO version     : " `cat $_prevASCDS_INSTALL/VERSION`
	    if ( -e "$_prevASCDS_INSTALL/VERSION.prop_tk" ) then
		echo " Proposal Toolkit version : " `cat "$_prevASCDS_INSTALL/VERSION".prop_tk`
	    endif
	    echo    " bin dir          : " "$_prevASCDS_INSTALL/bin"
	    echo    " Python path      : " "$CIAO_PYTHON"
	 endif
      endif
      echo "Please use the '-o' option to override the previous setup."
   endif
   exit
endif

################################################################
#       Remove previous configuration values from setup        #
################################################################

#  remove ASCDS_BIN from path if it exists 
if ( $?ASCDS_BIN ) then 
   if ( `echo "$path" | grep "$ASCDS_BIN"` != "" ) then
      set path = (`echo " $path " | sed "s% "$ASCDS_BIN" % %g"`)
   endif
endif

#  Remove ASCDS_INSTALL/lib/python3.11/site-packages from PYTHONPATH if it exists
if ( $?ASCDS_INSTALL ) then
  if ( $?PYTHONPATH ) then
    if ( `echo $PYTHONPATH | grep $ASCDS_INSTALL/ots/lib/python3.11/site-packages` != "" ) then
      setenv PYTHONPATH `echo "{$PYTHONPATH}" | sed "s%$ASCDS_INSTALL/ots/lib/python3.11/site-packages%%g"`
      setenv PYTHONPATH `echo "{$PYTHONPATH}" | sed "s%::%:%g"`
      setenv PYTHONPATH `echo "{$PYTHONPATH}" | sed "s%::%:%g"`
    endif
  endif
endif

#  Remove isis from path
if ( $?ASCDS_INSTALL ) then
   if ( `echo $path | grep $ASCDS_INSTALL/isis/bin` != "" ) then
	set path = (`echo " $path " | sed "s%$ASCDS_INSTALL/isis/bin% %g"`)
   endif
endif

#  remove $ASCDS_LIB from BWIDGET_LIBRARY if appropriate
#if ( $?ASCDS_LIB && $?BWIDGET_LIBRARY ) then 
#   if ( `echo $BWIDGET_LIBRARY | grep "$ASCDS_LIB"` != "" ) then
#      setenv BWIDGET_LIBRARY `echo "{$BWIDGET_LIBRARY}" | sed "s%$ASCDS_LIB%%g"`
#      setenv BWIDGET_LIBRARY `echo "{$BWIDGET_LIBRARY}" | sed "s%::%:%g"`
#   endif
#endif

#  remove $ASCDS_INSTALL/saord from BWIDGET_LIBRARY if appropriate
#if ( $?ASCDS_INSTALL && $?BWIDGET_LIBRARY ) then 
#   if ( `echo $BWIDGET_LIBRARY | grep "$ASCDS_INSTALL/saord"` != "" ) then
#      setenv BWIDGET_LIBRARY `echo "{$BWIDGET_LIBRARY}" | sed "s%$ASCDS_INSTALL/saord%%g"`
#      setenv BWIDGET_LIBRARY `echo "{$BWIDGET_LIBRARY}" | sed "s%::%:%g"`
#   endif
#endif

#  remove $ASCDS_INSTALL/Library/Frameworks from DYLD_FRAMEWORK_PATH if appropriate
#if ( $?ASCDS_INSTALL && $?DYLD_FRAMEWORK_PATH ) then 
#   if ( `echo "$DYLD_FRAMEWORK_PATH" | grep "$ASCDS_INSTALL/Library/Frameworks"` != "" ) then
#      setenv DYLD_FRAMEWORK_PATH `echo "{$DYLD_FRAMEWORK_PATH}" | sed "s%$ASCDS_INSTALL/Library/Frameworks%%g"`
#      setenv DYLD_FRAMEWORK_PATH `echo "{$DYLD_FRAMEWORK_PATH}" | sed "s%::%:%g"`
#   endif
#endif

#  if ASCDS_IMAGER_PATH is already in path, remove it
if ( "$?ASCDS_IMAGER_PATH" ) then 
   if ( `echo "$path" | grep "$ASCDS_IMAGER_PATH"` != "" ) then
      set path = (`echo " $path " | sed "s% $ASCDS_IMAGER_PATH % %g"`)
   endif
endif

################################################################
#                set CIAO environment variables                #
################################################################

setenv ASCDS_INSTALL "$_nextASCDS_INSTALL"
unset _nextASCDS_INSTALL
unset _nextCIAO_PYTHON
setenv ASCDS_BIN "$ASCDS_INSTALL/bin"
setenv ASCDS_OTS "$ASCDS_INSTALL"
setenv ASCDS_CONTRIB "$ASCDS_INSTALL"
setenv ASCDS_LIB "$ASCDS_INSTALL/lib"
setenv XPA_METHOD local

# CALDB environment variables 
setenv CALDB "$ASCDS_INSTALL/CALDB"
setenv CALDBCONFIG "$CALDB/software/tools/caldb.config"
setenv CALDBALIAS "$CALDB/software/tools/alias_config.fits"

# The calibration directory path
setenv ASCDS_CALIB "$ASCDS_INSTALL/data"
setenv ATOMDB "$ASCDS_INSTALL/ATOMDB"

#  The path of the working directory of the user
if ( ! $?ASCDS_WORK_PATH ) then
	if ( $?TMPDIR ) then
		setenv ASCDS_WORK_PATH $TMPDIR
	else
		setenv ASCDS_WORK_PATH /tmp
	endif
endif
if ( ! $?ASCDS_TMP ) then
	if ( $?TMPDIR ) then
		setenv ASCDS_TMP $TMPDIR
	else
		setenv ASCDS_TMP /tmp
	endif
endif

setenv ASCDS_SYS_PARAM "$ASCDS_INSTALL/param"

#  ahelp specific environment setup
alias about "${ASCDS_INSTALL}/bin/ahelp -k \!*"

if ( -f "$ASCDS_INSTALL/bin/ots_check" ) then
       set ASCDS_IMAGER_PATH = "`$ASCDS_INSTALL/bin/ots_check DS9`"
       if ( "$ASCDS_IMAGER_PATH" == "CIAO" ) then
	 if ( -d "/opt/conda/envs/ciao-4.17.0/imager" ) then
	    set ASCDS_IMAGER_PATH = "/opt/conda/envs/ciao-4.17.0/imager"
	 else 
	    set ASCDS_IMAGER_PATH = "$ASCDS_INSTALL/ots/bin"
	 endif
       endif
       setenv CIAO_XPA "`${ASCDS_INSTALL}/bin/ots_check XPA`"
       setenv CIAO_PYTHON "`$ASCDS_INSTALL/bin/ots_check PYTHON`"
       setenv CIAO_PYTHON_PATH "`$ASCDS_INSTALL/bin/ots_check PYTHON_PATH`"
       setenv CIAO_APP_PYTHON "`$ASCDS_INSTALL/bin/ots_check APP_PYTHON`"

       if ( "$CIAO_PYTHON" != "$ASCDS_INSTALL/ots/bin" && "$CIAO_PYTHON" != "CIAO" ) then
	   setenv CIAO_IPYTHON "`$ASCDS_INSTALL/bin/ots_check IPYTHON`"
	   setenv CIAO_APP_IPYTHON "`$ASCDS_INSTALL/bin/ots_check APP_IPYTHON`"
	   setenv CIAO_PYTHON_EXE "$CIAO_PYTHON/python"
       else
	   setenv CIAO_IPYTHON "CIAO"
	   setenv CIAO_APP_IPYTHON "CIAO"
	   setenv CIAO_PYTHON_EXE "$ASCDS_INSTALL/ots/bin/python"
       endif

       # copy .ciaorc file to Users $HOME directory if it dosn't exist.
       if ( ! -f "$HOME/.ciaorc" ) then
	    \cp "$ASCDS_INSTALL/config/system/.ciaorc" "$HOME"
	    \chmod 755 "$HOME/.ciaorc"
       endif
   else
       if ( ! $?ASCDS_IMAGER_PATH ) then
            if ( -d "/opt/conda/envs/ciao-4.17.0/imager" ) then
   	         set ASCDS_IMAGER_PATH = "/opt/conda/envs/ciao-4.17.0/imager"
            else
	         set ASCDS_IMAGER_PATH = "$ASCDS_INSTALL/ots/bin"
            endif
       endif
       setenv CIAO_PYTHON CIAO
       setenv CIAO_APP_PYTHON CIAO
       setenv CIAO_IPYTHON CIAO
       setenv CIAO_APP_IPYTHON CIAO
       setenv CIAO_PYTHON_PATH prepend

       if ( -f "/opt/conda/envs/ciao-4.17.0/bin" ) then
            setenv CIAO_XPA "/opt/conda/envs/ciao-4.17.0/bin"
       endif
endif

# ASCDS_IMAGER_PATH need to be a shell variable during
# setup but only an enviroment variable for CIAO to run.

setenv ASCDS_IMAGER_PATH "$ASCDS_IMAGER_PATH"
unset ASCDS_IMAGER_PATH

################################################################
#                   USER CUSTOMIZATIONS                        #
################################################################

################################################################
#                         Adjust paths                         # 
################################################################

#  Add ASCDS_BIN/ to path if it exists move to it the front
if ( `echo $path | grep $ASCDS_BIN` != "" ) then
   set path = (`echo " $path " | sed "s% $ASCDS_BIN % %g"`)
endif

if ( ${?path} ) then
   set path = ( "$ASCDS_BIN" $path )
else
   set path = ( "$ASCDS_BIN" )
endif

unset OS_REV scripts

# If we are running in CIAO mode, setup whatever else exists.

if ( ! $?ASCDS_CIAO ) then
   setenv ASCDS_CIAO "ciao"
endif

if ( $ASCDS_CIAO != "standalone" ) then
   if ( -e "$ASCDS_INSTALL/bin/setup_sherpa.csh" ) then
      source "$ASCDS_INSTALL/bin/setup_sherpa.csh"
   endif
   if ( -e "$ASCDS_INSTALL/bin/setup_tools.csh" ) then
      source "$ASCDS_INSTALL/bin/setup_tools.csh"
   endif
   if ( -e "$ASCDS_INSTALL/bin/setup_obsvis.csh" ) then
      source "$ASCDS_INSTALL/bin/setup_obsvis.csh"
   endif
endif

################################################################
#                    Finished configuration                    #  
################################################################

if ( $_silent == "No" && "$ASCDS_CIAO" == "ciao" ) then
   echo "CIAO configuration is complete... "
   if ( $_prevASCDS_INSTALL != ""  && $_partialConfig == "No" ) then
      echo "The CIAO setup for this window has changed from"
      if ( -e "$_prevASCDS_INSTALL/VERSION" ) then
	if ( -e "$_prevASCDS_INSTALL/bin/ciaover" ) then
	    $_prevASCDS_INSTALL/bin/ciaover -t $_prevASCDS_INSTALL
	else
	    echo " CIAO version     : " `cat "$_prevASCDS_INSTALL/VERSION"`
	    if ( -e $_prevASCDS_INSTALL/VERSION.prop_tk ) then
		echo " Proposal Toolkit version : " `cat $_prevASCDS_INSTALL/VERSION.prop_tk`
	    endif 
	    echo    " bin dir          : " "$_prevASCDS_INSTALL/bin"
	    echo    " Python path      : " "$_prevCIAO_PYTHON"
	endif
	echo "To:"
      endif
    endif
    if ( -e "$ASCDS_INSTALL/VERSION" ) then
      if ( -e "$ASCDS_INSTALL/bin/ciaover" ) then
	 $ASCDS_INSTALL/bin/ciaover -t $ASCDS_INSTALL
      else
	 echo " CIAO version     : " `cat "$ASCDS_INSTALL/VERSION"`
	 if ( -e $ASCDS_INSTALL/VERSION.prop_tk ) then
	    echo " Proposal Toolkit version : " `cat $ASCDS_INSTALL/VERSION.prop_tk`
	 endif 
	 echo    " bin dir          : " "$ASCDS_INSTALL/bin"
	 echo    " Python path      : " "$CIAO_PYTHON"
      endif
   endif
endif
unset _prevCIAO_PYTHON
unset _prevASCDS_INSTALL
unset ASCDS_CIAO
unsetenv ASCDS_CIAO
