#ifndef RDB_h
#define RDB_h

/* --8<--8<--8<--8<--
 *
 * Copyright (C) 2006 Smithsonian Astrophysical Observatory
 *
 * This file is part of RDB
 *
 * RDB is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * RDB is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the 
 *       Free Software Foundation, Inc. 
 *       51 Franklin Street, Fifth Floor
 *       Boston, MA  02110-1301, USA
 *
 * -->8-->8-->8-->8-- */

#include <fstream>
#include <iosfwd>
#include <iostream>
#include <string>
#include <vector>

#include <suplibxx/str.h>

#include <rdbxx/RDBErr.h>

#include <rdbxx/RDBComment.h>
#include <rdbxx/RDBColumnTmplt.h>

/// Provides interface for manipulating RDB tables
class RDB {
  /// @name Stream insertion and extraction operators.
  //@{
  /// Read table from input stream.
  friend std::istream& operator>>( std::istream& is, RDB& rdb );
  /// Write table to output stream.
  friend std::ostream& operator<<( std::ostream& os, RDB& rdb );
  //@}
  
 public:
  /// @name Enumerations for read/write/group return status.
  //@{
  /// Acceptable column justifications.
  enum Status { REOF=0x00, REOL=0x01, REOG=0x02, RBOG=0x04 };
  //@}

  /// @name Constructing and destructing and initializing RDB objects.
  //@{
  /// Attaches RDB object to a file.
  RDB( const std::string& name="", std::ios::openmode mode=std::ios::in );
  /// Attaches RDB object to an istream.
  RDB( std::istream* isptr );
  /// Attaches RDB object to an ostream.
  RDB( std::ostream* osptr );
  /// Copies RDB object.
  RDB( const RDB& rdb );
  /// Deletes resources allocated by the RDB object.
  ~RDB( void );
  //@}
  
  /// @name I/O related operations.
  //@{
  /// Attaches RDB object to a file.
  void open( const std::string& name, std::ios::openmode mode=std::ios::in );
  /// Attaches RDB object to an istream.
  void open( std::istream* isptr );
  /// Attaches RDB object to an ostream.
  void open( std::ostream* osptr );
  /// Copies RDB object.
  void open( const RDB& rdb );
  /// Closes the stream attached to RDB object.
  void close( void );
  
  ///Read a line of data from the istream.
  int read( void );
  /// Write a line of data to the ostream.
  bool write( void );
  /// Rewind the stream to the beginning of the first row of data.
  bool rewind( void );
  //@}

  /// @name Auto-indexing related methods.
  //@{
  /// Indicates if auto-indexing is activated.
  bool autoIdx( void ) const;
  /// Activates/deactivates auto-indexing.
  void autoIdx( const bool on );
  /// Increments the indices in the RDBColumn data elements.
  void advanceIdx( void );
  //@}

  /// @name Column group manipulation (break columns)
  //@{
  /// Turn on/off group status for the named column.
  void setGroup( const std::string& name, bool group=true );
  /// Turn on/off group status for the indexed column.
  void setGroup( const int idx, bool group=true );
  /// Returns group status, true if its a new group, for the named column.
  bool getGroup( const std::string& name );
  /// Returns group status, true if its a new group, for the indexed column.
  bool getGroup( const int idx );
  /// Checks if any column indicates a new group.
  bool newGroup( void );
  //@}
  
  /// @name Comment accessors.
  //@{
  /// Add RDBComment in header of RDB object. 
  void setComment( const std::string& comm, const int idx=-1 );
  /// Add or replace RDBComment in header of RDB object.
  void setComment( RDBComment& comm, const int idx=-1 );
  /// Add or replace RDBComment in header of RDB object.
  void setComment( RDBComment& comm, const std::string& name, const size_t idx=0 );
  /// Copy all comments from an existing RDB object.
  void setComment( const RDB& rdb );
  
  /// Return RDBComment at given index.
  RDBComment& getComment( const size_t idx );
  /// Return RDBComment with given keyword.
  RDBComment& getComment( const std::string& name, const size_t idx=0 );
  //@}
  
  /// @name Column accessors.
  //@{
  /// Add an RDBColumn in RDB object.
  void setColumn( const std::string& name, const std::string& def, const int idx=-1 );
  /// Add or replace RDBColumn in RDB object.
  void setColumn( RDBColumn* col, const int idx=-1 );
  /// Add of replace RDBColumn in RDB object.
  void setColumn( RDBColumn* col, const std::string& name, const size_t idx=0 );
  /// Copy all columns from an existing RDB object.
  void setColumn( const RDB& rdb );
  
  /// Return pointer to RDBColumn at given index.
  RDBColumn* getColumn( const size_t idx );
  /// Return pointer to RDBColumn with given name.
  RDBColumn* getColumn( const std::string& name, const size_t idx=0 );
  //@}

  /// @name Column index based accessors.
  //@{
  /// Modify the name of the RDBColumn at idx.
  void setName( const size_t idx, const std::string& name );
  /// Modify the definition of the RDBColumn at idx.
  void setDef( const size_t idx, const std::string& def  );
  /// Modify the width of the RDBColumn at idx.
  void setWidth( const size_t idx, const long width );
  /// Modify the type of the RDBColumn at idx.
  void setType( const size_t idx, const RDBColumn::Type type );
  /// Modify the justification of the RDBColumn at idx.
  void setJust( const size_t idx, const RDBColumn::Just just );
  /// Modify the description of the RDBColumn at idx.
  void setDesc( const size_t idx, const std::string& desc );
  /// Map RDBColumn data to user-supplied memory.
  void mapData( const size_t idx, double data[], const size_t nelems=1 );
  /// Map RDBColumn data to user-supplied memory.
  void mapData( const size_t idx, long data[], const size_t nelems=1 );
  /// Map RDBColumn data to user-supplied memory.
  void mapData( const size_t idx, std::string data[], const size_t nelems=1 );
  /// Sets the data value of RDBColumn, converting as necessary.
  void setData( const size_t idx, const double data );
  /// Sets the data value of RDBColumn, converting as necessary.
  void setData( const size_t idx, const long data );
  /// Sets the data value of RDBColumn, converting as necessary.
  void setData( const size_t idx, const std::string& data );

  /// Return the name of the RDBColumn at idx.
  void getName( const size_t idx, std::string& name ) const;
  /// Return the definition of the RDBColumn at idx.
  void getDef( const size_t idx, std::string& def  );
  /// Return the width of the RDBColumn at idx.
  void getWidth( const size_t idx, long& width ) const;
  /// Return the type of the RDBColumn at idx.
  void getType( const size_t idx, RDBColumn::Type& type ) const;
  /// Return the just of the RDBColumn at idx.
  void getJust( const size_t idx, RDBColumn::Just& just ) const;
  /// Return the description of the RDBColumn at idx.
  void getDesc( const size_t idx, std::string& desc ) const;
  /// Return the data of the RDBColumn at idx, converting if necessary.
  void getData( const size_t idx, double& data );
  /// Return the data of the RDBColumn at idx, converting if necessary.
  void getData( const size_t idx, long& data );
  /// Return the data of the RDBColumn at idx, converting if necessary.
  void getData( const size_t idx, std::string& data );

  /// Return the name of the RDBColumn at idx.
  std::string getName( const size_t idx ) const;
  /// Return the definition of the RDBColumn at idx.
  std::string getDef( const size_t idx  );
  /// Return the width of the RDBColumn at idx.
  long getWidth( const size_t idx ) const;
  /// Return the type of the RDBColumn at idx.
  RDBColumn::Type getType( const size_t idx ) const;
  /// Return the justification of the RDBColumn at idx.
  RDBColumn::Just getJust( const size_t idx ) const;
  /// Return the description of the RDBColumn at idx.
  std::string getDesc( const size_t idx ) const;
  /// Return the data of the RDBColumn at idx, converting if necessary.
  double getDataDouble( const size_t idx );
  /// Return the data of the RDBColumn at idx, converting if necessary.
  long getDataLong( const size_t idx );
  /// Return the data of the RDBColumn at idx, converting if necessary.
  std::string getDataString( const size_t idx );
  //@}

  /// @name Column name based accessors.
  //@{
  /// Modify the RDBColumn name.
  void setName( const std::string& name, const std::string& newname );
  /// Modify the RDBColumn definition.
  void setDef( const std::string& name, const std::string& def  );
  /// Modify the RDBColumn width.
  void setWidth( const std::string& name, const long width );
  /// Modify the RDBColumn type.
  void setType( const std::string& name, const RDBColumn::Type type );
  /// Modify the RDBColumn justification.
  void setJust( const std::string& name, const RDBColumn::Just just );
  /// Modify the RDBColumn description.
  void setDesc( const std::string& name, const std::string& desc );
  /// Map RDBColum data to user-supplied memory.
  void mapData( const std::string& name, double data[], const size_t nelems=1 );
  /// Map RDBColum data to user-supplied memory.
  void mapData( const std::string& name, long data[], const size_t nelems=1 );
  /// Map RDBColum data to user-supplied memory.
  void mapData( const std::string& name, std::string data[], const size_t nelems=1 );
  /// Modify the RDBColumn data, converting if necessary.
  void setData( const std::string& name, const double data );
  /// Modify the RDBColumn data, converting if necessary.
  void setData( const std::string& name, const long data );
  /// Modify the RDBColumn data, converting if necessary.
  void setData( const std::string& name, const std::string& data );

  /// Return the name of the RDBColumn.
  void getName( const std::string& name, std::string& namefound ) const;
  /// Return the definition of the RDBColumn.
  void getDef( const std::string& name, std::string& def  );
  /// Return the width of the RDBColumn.
  void getWidth( const std::string& name, long& width ) const;
  /// Return the type of the RDBColumn.
  void getType( const std::string& name, RDBColumn::Type& type ) const;
  /// Return the justification of the RDBColumn.
  void getJust( const std::string& name, RDBColumn::Just& just ) const;
  /// Return the description of the RDBColumn.
  void getDesc( const std::string& name, std::string& desc ) const;
  /// Return the data of the RDBColumn, converting if necessary.
  void getData( const std::string& name, double& data );
  /// Return the data of the RDBColumn, converting if necessary.
  void getData( const std::string& name, long& data );
  /// Return the data of the RDBColumn, converting if necessary.
  void getData( const std::string& name, std::string& data );

  /// Return the name of the RDBColumn.
  std::string getName( const std::string& name ) const;
  /// Return the definition of the RDBColumn.
  std::string getDef( const std::string& name  );
  /// Return the width of the RDBColumn.
  long getWidth( const std::string& name ) const;
  /// Return the type of the RDBColumn.
  RDBColumn::Type getType( const std::string& name ) const;
  /// Return the justification of the RDBColumn.
  RDBColumn::Just getJust( const std::string& name ) const;
  /// Return the description of the RDBColumn.
  std::string getDesc( const std::string& name ) const;
  /// Return the data of the RDBColumn, converting if necessary.
  double getDataDouble( const std::string& name );
  /// Return the data of the RDBColumn, converting if necessary.
  long getDataLong( const std::string& name );
  /// Return the data of the RDBColumn, converting if necessary.
  std::string getDataString( const std::string& name );
  //@}
  
  /// @name Table and header statistics.
  //@{
  /// Return number of comments in RDB object.
  size_t nComments( void ) const;
  /// Return number of columns in RDB object.
  size_t nColumns( void ) const;
  /// Return number of rows in RDB object.
  size_t nRows( void ); // Not const since it requires scanning the entire file...
  //@}
  
 protected:
  /// Parse header, i.e. comments and column names and definitions.
  void parseHeader( void );
  /// Parse fields in a row.
  std::vector<std::string> parseLine( const std::string& line ) const;
  /// Parse fields in a row.
  size_t parseLine( bool& newgroup );

  /// Name of RDB file.
  std::string _filename;
  /// Open mode of the associated stream.
  std::ios::openmode _mode;

  /// Istream attached to data file.
  std::istream* _isptr;
  /// Ostream attached to data file.
  std::ostream* _osptr;
  /// Indicates if RDB object is responsible for deallocating the istream.
  bool _myisptr;
  /// Indicates if RDB object is responsible for deallocating the ostream.
  bool _myosptr; 
  
  /// Position of beginning of first row of data.
  size_t _rewindto;
  /// Number of comments.
  size_t _ncomms;
  /// Number of columns.
  size_t _ncols;
  /// Number of rows.
  size_t _nrows;
  /// Indicates if associated file must be scanned to determine number of rows.
  bool _knowrows;
  /// Current *table* row number. 
  long _rownum;
  /// Current *file* row number.
  long _frownum;
  
  /// Indicates if RDBColumn data elements should be advanced.
  bool _autoidx;
  /// Indicates if this is the first call to RDB::read.
  bool _firstread;
  /// Indicates if this is the last call to RDB::read.
  bool _lastread;
  /// Indicates if the header has been output.
  bool _writehdr;

  /// Array of RDBComments.
  RDBComment* _comms;
  /// Array of RDBColumns.
  RDBColumn** _cols;
  /// Hidden column, containing row number.
  RDBLongColumn _nrcol;
  /// Indicates if RDB object is responsible for deallocating given RDBColumn.
  bool* _mycols;

  /// Line from RDB table.
  std::string _line;
  
  
  /// Line from RDB table.
    //  char* _line;
  /// Number of characters in the line buffer.
    //  size_t _linesize;
  
};

#endif

